package ab.issue.model.values;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Embedded;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.ForeignKey;
import android.arch.persistence.room.Index;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.support.annotation.NonNull;

import com.google.common.base.MoreObjects;
import com.google.firebase.ml.vision.barcode.FirebaseVisionBarcode;
import com.google.zxing.client.result.EmailAddressParsedResult;

import java.util.List;
import java.util.Locale;

import javax.annotation.Nonnull;

import ab.issue.db.BarcodeDatabase;
import ab.issue.db.DbConstants;
import ab.issue.model.Barcode;
import androidx.navigation.NavController;

import static com.google.common.base.Optional.fromNullable;

@SuppressWarnings("Guava")
@Entity(
        tableName = DbConstants.EMAIL_TABLE_NAME,
        foreignKeys = @ForeignKey(
                entity = Barcode.class,
                parentColumns = DbConstants.COLUMN_NAME_BARCODE_ID,
                childColumns = DbConstants.COLUMN_NAME_EMAIL_BARCODE_ID,
                onUpdate = ForeignKey.CASCADE,
                onDelete = ForeignKey.CASCADE
        ),
        indices = @Index(value = DbConstants.COLUMN_NAME_EMAIL_BARCODE_ID, unique = true)
)
public class EmailBarcodeFormat implements BarcodeFormatValue {

    @PrimaryKey(autoGenerate = true)
    @ColumnInfo(name = DbConstants.COLUMN_NAME_EMAIL_ID)
    private long id;

    @ColumnInfo(name = DbConstants.COLUMN_NAME_EMAIL_BARCODE_ID)
    private long barcodeId;

    @Nonnull
    @Embedded
    private Email email;

    public EmailBarcodeFormat(@NonNull Email email) {
        this.email = email;
    }

    public static EmailBarcodeFormat create(FirebaseVisionBarcode.Email email) {
        return new EmailBarcodeFormat(Email.create(email));
    }

    public static EmailBarcodeFormat create(EmailAddressParsedResult emailResult) {
        return new EmailBarcodeFormat(Email.create(emailResult));
    }

    @Override
    public String getName(Context context) {
        return null;
    }

    @Override
    public BarcodeFormatValueType getType() {
        return BarcodeFormatValueType.EMAIL;
    }

    @Override
    public String encode() {
        return String.format(Locale.ENGLISH, "mailto:%s?subject=%s&body=%s",
                fromNullable(email.getAddress()).or(""),
                fromNullable(email.getSubject()).or(""),
                fromNullable(email.getBody()).or(""));
    }

    @NonNull
    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("email", email)
                .toString();
    }

    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    @Override
    public void setBarcodeId(long barcodeId) {
        this.barcodeId = barcodeId;
    }

    public long getBarcodeId() {
        return barcodeId;
    }

    @Override
    public int getIcon() {
        return 0;
    }

    @Override
    public void save(BarcodeDatabase barcodeDatabase) {
        barcodeDatabase.emailDao().insert(this);
    }

    @Override
    public void create(NavController navController) {
//        navController.navigate(R.id.action_create_email_barcode);
    }

    @Override
    public List<Barcode> search(BarcodeDatabase barcodeDatabase, String query) {
        return barcodeDatabase.emailDao().findTextInBarcodeFormatValues(query);
    }

    @NonNull
    public Email getEmail() {
        return email;
    }


    public void setEmail(@NonNull Email email) {
        this.email = email;
    }
}

