package ab.issue.model.values;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.ForeignKey;
import android.arch.persistence.room.Index;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.common.base.MoreObjects;
import com.google.firebase.ml.vision.barcode.FirebaseVisionBarcode;
import com.google.zxing.client.result.GeoParsedResult;

import java.util.List;
import java.util.Locale;

import ab.issue.db.BarcodeDatabase;
import ab.issue.db.DbConstants;
import ab.issue.model.Barcode;
import androidx.navigation.NavController;

@Entity(
        tableName = DbConstants.GEO_TABLE_NAME,
        foreignKeys = @ForeignKey(
                entity = Barcode.class,
                parentColumns = DbConstants.COLUMN_NAME_BARCODE_ID,
                childColumns = DbConstants.COLUMN_NAME_GEO_BARCODE_ID,
                onUpdate = ForeignKey.CASCADE,
                onDelete = ForeignKey.CASCADE
        ),
        indices = @Index(value = DbConstants.COLUMN_NAME_GEO_BARCODE_ID, unique = true)
)
public class GeoBarcodeFormat implements BarcodeFormatValue {

    public static GeoBarcodeFormat create(@Nullable FirebaseVisionBarcode.GeoPoint geoPoint) {
        GeoBarcodeFormat result = new GeoBarcodeFormat();
        if (geoPoint != null) {
            result.latitude = geoPoint.getLat();
            result.longitude = geoPoint.getLng();
        }
        return result;
    }

    public static GeoBarcodeFormat create(GeoParsedResult geoParsedResult) {
        GeoBarcodeFormat result = new GeoBarcodeFormat();
        if (geoParsedResult != null) {
            result.longitude = geoParsedResult.getLongitude();
            result.latitude = geoParsedResult.getLatitude();
        }
        return result;
    }

    @PrimaryKey(autoGenerate = true)
    @ColumnInfo(name = DbConstants.COLUMN_NAME_GEO_ID)
    private long id;

    @ColumnInfo(name = DbConstants.COLUMN_NAME_GEO_BARCODE_ID)
    private long barcodeId;

    @ColumnInfo(name = DbConstants.COLUMN_NAME_GEO_LATITUDE)
    private double latitude;

    @ColumnInfo(name = DbConstants.COLUMN_NAME_GEO_LONGITUDE)
    private double longitude;

    @Override
    public String getName(Context context) {
        return null;
    }

    @Override
    public BarcodeFormatValueType getType() {
        return BarcodeFormatValueType.GEO;
    }

    @Override
    public String encode() {
        return String.format(Locale.ENGLISH, "geo:%f,%f", latitude, longitude);
    }

    @NonNull
    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("latitude", latitude)
                .add("longitude", longitude)
                .omitNullValues()
                .toString();
    }

    @Override
    public int getIcon() {
        return 0;
    }

    @Override
    public void create(NavController navController) {
//        navController.navigate(R.id.action_create_geo_barcode);
    }

    @Override
    public void setBarcodeId(long barcodeId) {
        this.barcodeId = barcodeId;
    }

    @Override
    public void save(BarcodeDatabase barcodeDatabase) {
        barcodeDatabase.geoDao().insert(this);
    }

    @Override
    public List<Barcode> search(BarcodeDatabase barcodeDatabase, String query) {
        return barcodeDatabase.geoDao().findTextInBarcodeFormatValues(query);
    }

    public long getBarcodeId() {
        return barcodeId;
    }

    public void setId(long id) {
        this.id = id;
    }

    public long getId() {
        return id;
    }

    public double getLatitude() {
        return latitude;
    }

    public void setLatitude(double latitude) {
        this.latitude = latitude;
    }

    public double getLongitude() {
        return longitude;
    }

    public void setLongitude(double longitude) {
        this.longitude = longitude;
    }
}

