package ab.issue.model.values;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Embedded;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.ForeignKey;
import android.arch.persistence.room.Index;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;

import com.google.common.base.MoreObjects;
import com.google.common.collect.Lists;
import com.google.firebase.ml.vision.barcode.FirebaseVisionBarcode;
import com.google.zxing.client.result.TelParsedResult;

import java.util.List;
import java.util.Locale;

import javax.annotation.Nonnull;

import ab.issue.R;
import ab.issue.db.BarcodeDatabase;
import ab.issue.db.DbConstants;
import ab.issue.model.Barcode;
import androidx.navigation.NavController;

import static com.google.common.base.Optional.fromNullable;

@SuppressWarnings("Guava")
@Entity(
        tableName = DbConstants.PHONE_TABLE_NAME,
        foreignKeys = @ForeignKey(
                entity = Barcode.class,
                parentColumns = DbConstants.COLUMN_NAME_BARCODE_ID,
                childColumns = DbConstants.COLUMN_NAME_PHONE_BARCODE_ID,
                onUpdate = ForeignKey.CASCADE,
                onDelete = ForeignKey.CASCADE
        ),
        indices = @Index(value = DbConstants.COLUMN_NAME_PHONE_BARCODE_ID, unique = true)
)
public class PhoneBarcodeFormat implements BarcodeFormatValue {

    public static PhoneBarcodeFormat create(@Nullable FirebaseVisionBarcode.Phone phone) {
        return phone != null
                ? new PhoneBarcodeFormat(Phone.create(phone))
                : new PhoneBarcodeFormat(new Phone());
    }

    public static PhoneBarcodeFormat create(TelParsedResult telParsedResult) {
        return new PhoneBarcodeFormat(Phone.create(telParsedResult));
    }

    @PrimaryKey(autoGenerate = true)
    @ColumnInfo(name = DbConstants.COLUMN_NAME_PHONE_ID)
    private long id;

    @ColumnInfo(name = DbConstants.COLUMN_NAME_PHONE_BARCODE_ID)
    private long barcodeId;

    @Nonnull
    @Embedded
    private final Phone phone;

    public PhoneBarcodeFormat(@NonNull Phone phone) {
        this.phone = phone;
    }

    @Override
    public String getName(Context context) {
        return null;
    }

    @Override
    public BarcodeFormatValueType getType() {
        return BarcodeFormatValueType.PHONE;
    }

    @Override
    public String encode() {
        return String.format(Locale.ENGLISH, "tel:%s", fromNullable(phone.getNumber()).or(""));
    }

    @NonNull
    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("phone", phone)
                .toString();
    }

    @Override
    public int getIcon() {
        return 0;
    }

    @Override
    public void save(BarcodeDatabase barcodeDatabase) {
        barcodeDatabase.phoneDao().insert(this);
    }

    @Override
    public void setBarcodeId(long barcodeId) {
        this.barcodeId = barcodeId;
    }

    @Override
    public void create(NavController navController) {
//        navController.navigate(R.id.action_create_phone_barcode);
    }

    @Override
    public List<Barcode> search(BarcodeDatabase barcodeDatabase, String query) {
        return barcodeDatabase.phoneDao().findTextInBarcodeFormatValues(query);
    }

    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    public long getBarcodeId() {
        return barcodeId;
    }

    @Nonnull
    public Phone getPhone() {
        return phone;
    }
}

