package ab.issue.model.values;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Embedded;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.ForeignKey;
import android.arch.persistence.room.Index;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.common.base.MoreObjects;
import com.google.firebase.ml.vision.barcode.FirebaseVisionBarcode;
import com.google.zxing.client.result.URIParsedResult;

import java.util.List;
import java.util.Locale;

import javax.annotation.Nonnull;

import ab.issue.db.BarcodeDatabase;
import ab.issue.db.DbConstants;
import ab.issue.model.Barcode;
import androidx.navigation.NavController;

@Entity(
        tableName = DbConstants.URL_TABLE_NAME,
        foreignKeys = @ForeignKey(
                entity = Barcode.class,
                parentColumns = DbConstants.COLUMN_NAME_BARCODE_ID,
                childColumns = DbConstants.COLUMN_NAME_URL_BARCODE_ID,
                onUpdate = ForeignKey.CASCADE,
                onDelete = ForeignKey.CASCADE
        ),
        indices = @Index(value = DbConstants.COLUMN_NAME_URL_BARCODE_ID, unique = true)
)
public class UrlBarcodeFormat implements BarcodeFormatValue {

    public static UrlBarcodeFormat create(@Nullable FirebaseVisionBarcode.UrlBookmark urlBookmark) {
        return new UrlBarcodeFormat(UrlBookmark.create(urlBookmark));
    }

    public static UrlBarcodeFormat create(URIParsedResult uriParsedResult) {
        return new UrlBarcodeFormat(UrlBookmark.create(uriParsedResult));
    }

    @PrimaryKey(autoGenerate = true)
    @ColumnInfo(name = DbConstants.COLUMN_NAME_URL_ID)
    private long id;

    @ColumnInfo(name = DbConstants.COLUMN_NAME_URL_BARCODE_ID)
    private long barcodeId;

    @Nonnull
    @Embedded
    private final UrlBookmark urlBookmark;

    public UrlBarcodeFormat(@NonNull UrlBookmark urlBookmark) {
        this.urlBookmark = urlBookmark;
    }

    @Override
    public String getName(Context context) {
        return null;
    }

    @Override
    public BarcodeFormatValueType getType() {
        return BarcodeFormatValueType.URL;
    }

    @NonNull
    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("urlBookmark", urlBookmark)
                .add("type", getType())
                .toString();
    }

    @Override
    public String encode() {
        return String.format(Locale.ENGLISH, "MEBKM:TITLE:%s;URL:%s;;", urlBookmark.getTitle(), urlBookmark.getUrl());
    }

    @Override
    public int getIcon() {
        return 0;
    }

    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    @Override
    public void setBarcodeId(long barcodeId) {
        this.barcodeId = barcodeId;
    }

    @Override
    public void save(BarcodeDatabase barcodeDatabase) {
        barcodeDatabase.urlDao().insert(this);
    }

    @Override
    public void create(NavController navController) {
//        navController.navigate(R.id.action_create_url_barcode);
    }

    @Override
    public List<Barcode> search(BarcodeDatabase barcodeDatabase, String query) {
        return barcodeDatabase.urlDao().findTextInBarcodeFormatValues(query);
    }

    @Nonnull
    public UrlBookmark getUrlBookmark() {
        return urlBookmark;
    }

    public long getBarcodeId() {
        return barcodeId;
    }
}

