package dominando.android.presentation

import androidx.arch.core.executor.testing.InstantTaskExecutorRule
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import com.nhaarman.mockitokotlin2.any
import com.nhaarman.mockitokotlin2.mock
import com.nhaarman.mockitokotlin2.verify
import com.nhaarman.mockitokotlin2.whenever
import dominando.android.data.BooksRepository
import dominando.android.data.model.Book
import dominando.android.data.model.MediaType
import dominando.android.data.model.Publisher
import dominando.android.domain.ListBooksUseCase
import dominando.android.domain.RemoveBookUseCase
import dominando.android.domain.ViewBookDetailsUseCase
import dominando.android.presentation.binding.BookConverter
import dominando.android.presentation.binding.Book as BookBinding
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import java.util.*

class ListBooksViewModelTest {
    @get:Rule
    val rule = InstantTaskExecutorRule()

    private val dummyBooksList = listOf(
            Book().apply {
                id = UUID.randomUUID().toString()
                title = "Dominando o Android"
                author = "Nelson Glauber"
                available = true
                coverUrl = ""
                pages = 954
                publisher = Publisher(UUID.randomUUID().toString(), "Novatec")
                year = 2018
                mediaType = MediaType.PAPER
                rating = 5f
            },
            Book().apply {
                id = UUID.randomUUID().toString()
                title = "Clean Code"
                author = "Uncle Bob"
                available = true
                coverUrl = ""
                pages = 465
                publisher = Publisher(UUID.randomUUID().toString(), "Outro")
                year = 2009
                mediaType = MediaType.EBOOK
                rating = 5f
            }
    )

    val repository: BooksRepository = mock()

    @Before
    fun init() {
        whenever(repository.loadBooks())
                .thenReturn(
                        MutableLiveData<List<Book>>().apply {
                            value = dummyBooksList
                        }
                )
        whenever(repository.remove(any()))
                .thenReturn(
                        MutableLiveData<Boolean>().apply {
                            value = true
                        }
                )
    }

    @Test
    fun testBookDetailsIsLoaded() {
        // Given
        val viewModel = BookListViewModel(ListBooksUseCase(repository), RemoveBookUseCase(repository))
        val observer = mock<Observer<List<BookBinding>>>()
        // When
        viewModel.getBooks().observeForever(observer)
        // Then
        verify(observer).onChanged(dummyBooksList.map { book -> BookConverter.fromData(book) })
    }
}