package com.ingenico.pclservice;

/* Import our Parcelable message */
import com.ingenico.pclservice.TransactionIn;
import com.ingenico.pclservice.TransactionOut;

import com.ingenico.pclservice.IPclServiceCallback;
 
// PCL service interface.
interface IPclService {
    /**
     * Gets server status. 
     *
     * @param result  server status (0x10 means CONNECTED, 0x20 means DISCONNECTED)
     * @return      always true
     */
    boolean serverStatus(out byte[] result);
    /**
     * Open printer on remote device. 
     *
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean openPrinter(out byte[] result); 
	/**
     * Close printer on remote device. 
     *
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean closePrinter(out byte[] result);
	/**
     * Print text on remote device. 
     *
     * @param strText  Null-terminated string with text to be printed. Max length is 511 characters. 
     * The string must end with line feed character.
     * @param result   Result of the operation (0=OK, 1=KO)
     * @return         true if the method succeeded (see result for command result).
     *                 false if the method failed.
     */
	boolean printText(String strText, out byte[] result);
	/**
     * Print bitmap on remote device.
     * Bitmap characteristics are:
     * -   max width: 384
     * -   max height: 1024
     * -   no compression
     * -   1, 2 or 4 color bits 
     *
     * @param bmpBuf  Bitmap buffer
     * @param bmpSize Bitmap size
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean printBitmap(in byte[] bmpBuf, int bmpSize, out byte[] result);
	/**
     * Store logo (bitmap image) in remote device for further printing.
     * Bitmap characteristics are:
     * -   max width: 384
     * -   max height: 1024
     * -   no compression
     * -   1, 2 or 4 color bits
     *
     * @param name    Null-terminated string containing the name of the logo
     * (8 characters max, upper case)
     * @param type    Unused. Must be set to 1.
     * @param bmpBuf  Bitmap buffer
     * @param bmpSize Bitmap size
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean storeLogo(String name, int type, in byte[] bmpBuf, int bmpSize, out byte[] result);
	/**
     * Print logo (bitmap image) stored in iWL (either with previous call to 
     * storeLogo or with other available methods used to load some files in 
     * terminal: LLT, Ingestate).
     *
     * @param name    Null-terminated string containing the name of the logo 
     * (8 characters max, upper case)
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean printLogo(String name, out byte[] result);
	/**
     * Get remote device printer status.
     *
     * @param result  Result of the operation (bitfield)
     *                0x00=OK
     *                0x01=KO
     *                0x10=PRINTER ERROR
     *                0x20=PAPER OUT
     *                0x40=CONNECTION FAILED (when using Bluetooth printer)
     *                0x80=PRINTER BATTERY LOW (when using Bluetooth printer)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean getPrinterStatus(out byte[] result);
	/**
     * Get remote device time.
     *
     * @param time    Time of the remote device.
     * @return        true if the method succeeded (see time for command result).
     *                false if the method failed.
     */
	boolean getTerminalTime(out byte[] time);
	/**
     * Set remote device time.
     *
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean setTerminalTime(out byte[] result);
	/**
     * Read Telium information (serial number, product number)
     *
     * @param serialNumber  Serial number of the remote device.
     * @param productNumber  Product number of the remote device.
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean getTerminalInfo(out byte[] serialNumber, out byte[] productNumber);
	/**
     * Fetch information about the components installed on the Telium terminal.
     *
     * @param strPath  File name where to store components list.
     * The file will be stored in PclService data directory.
     * You can access it with:
     * createPackageContext("com.ingenico.pclservice",0).openFileInput(strPath);
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean getTerminalComponents(String strPath);
	/**
     * This launches a transaction request, passing the input parameters to the payment module.
     * The output parameters are filled with the response from the payment module.
     *
     * @param transIn  Input parameters for transaction.
     * @param transOut Output parameters for transaction.
     * @return        true if the method succeeded
     *                false if the method failed.
     * @see TransactionIn
     * @see TransactionOut
     */
	boolean doTransaction(in TransactionIn transIn, inout TransactionOut transOut);
	/**
     * This launches a transaction request, passing the input parameters to the payment module
     * and custom extended input data. The output parameters are filled with the response from 
     * the payment module and custom extended output data are received too.
     *
     * @param transIn  Input parameters for transaction.
     * @param transOut Output parameters for transaction.
     * @param appNumber This value explicitly specified the Telium application number to contact
     * (destination of the PDA_START_TRANSACTION ServiceCall, see Telium documentation for more detail).
     * Set it to zero if no explicit selection is required.
     * @param inBuffer      Pointer to the input extended data to pass as parameter to the PDA_START_TRANSACTION ServiceCall.
     * @param inBufferSize  Size of the input extended data to pass as parameter to the PDA_START_TRANSACTION ServiceCall.
     * @param outBuffer     Pointer to the output extended data to receive as parameter from the PDA_START_TRANSACTION ServiceCall.
     * @param outBufferSize Maximum size of the output extended data to receive as parameter from the PDA_START_TRANSACTION ServiceCall, 
     * and this variable is updated with the actual buffer size.
     * @return        true if the method succeeded
     *                false if the method failed.
     * @see TransactionIn
     * @see TransactionOut
     */
	boolean doTransactionEx(in TransactionIn transIn, inout TransactionOut transOut, int appNumber, in byte[] inBuffer, long inBufferSize, inout byte[] outBuffer, inout long[] outBufferSize);
	/**
     * Launch a Telium update.
     *
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean doUpdate(inout byte[] result);
	/**
     * Reset Telium terminal.
     *
     * @param info  code to return when resetting the terminal (see OEM_exit() definition 
     * in Telium SDK Help for more information on return code).
     * @return        true if the method succeeded
     *                false if the method failed.
     */
	boolean resetTerminal(in int info);
	/**
     * Simulate input key on the Telium terminal.
     *
     * @param key  string (null terminated) containing the input to simulate 
     * (only valid Telium input are allowed, see Telium SDK Help for more information).
     * @return        true if the method succeeded
     *                false if the method failed.
     */
	boolean inputSimul(in byte[] key);
	/**
     * Send message to the terminal.
     *
     * @param message  Buffer containing the message to send to the terminal.
     * (size must not be greater than 1024 bytes)
     * @param byteSent Number of bytes sent, if the message cannot be sent the result is equal to zero
     * @return        true if the method succeeded
     *                false if the method failed.
     */
	boolean sendMessage(in byte[] message, inout int[] byteSent);
	/**
     * Receive a message from the terminal.
     *
     * @param message  Buffer that will receive the message received. 
     * @param byteRecvd Number of bytes received, if no message received, the value returned in this variable is zero.
     * @return        true if the method succeeded
     *                false if the method failed.
     */
	boolean receiveMessage(inout byte[] message, inout int[] byteRecvd);
	/**
     * Flush all previous messages received from the terminal.
     *
     * @return        true if the method succeeded
     *                false if the method failed.
     */
	boolean flushMessages();
	/**
     * Launch a M2OS shortcut.
     *
     * @param shortcut  string (null terminated) containing the number of the shortcut. 
     * @return        true if the method succeeded
     *                false if the method failed.
     */
	boolean launchM2OSShortcut(in byte[] shortcut);
	/**
     * Sets signature capture result. 
     *
     * @param result  0 if signature capture is successful
     */
	void setSignatureCaptureResult(int result);
	/**
     * Open barcode reader on remote device. 
     *
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
	boolean openBarcode(out byte[] result); 
	/**
     * Close barcode reader on remote device. 
     *
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
    boolean closeBarcode(out byte[] result);
    //boolean getFirmwareVersion(out byte[] fwVersion);
    /**
     * Add TCP/IP bridge between Android device and Telium device. 
     *
     * @param port		TCP port to connect to or to listen (depending on direction parameter)
     * @param direction Bridge direction: 0 means Android connects to Telium TCP port
     *                                    1 means Telium connects to Android TCP port
     */
    void addDynamicBridge(in int port, in int direction);
    /**
     * Get printer battery status. It is recommended to load the printer battery before printing
     * if the battery is low.
     *
     * @param result  Result of the operation (1 means battery is low, 0 means battery is not low)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
    boolean isPrinterBatteryLow(out byte[] result);
    /**
     * Get printer connection status. (for Bluetooth printers).
     *
     * @param result  Result of the operation (1 means connected, 0 means not connected)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
    boolean isPrinterConnected(out byte[] result);
    /**
     * Registers a callback interface with the service.
     */
    void registerCallback(IPclServiceCallback cb, boolean externalSignCap);
    
    /**
     * Removes a previously registered callback interface.
     */
    void unregisterCallback(IPclServiceCallback cb);
    
    /**
     * Submits signature capture image as Bitmap. 
     *
     * @param bmp  Bitmap representing signature or null if signature capture failed.
     */
    void submitSignatureWithImage(in Bitmap bmp);
    
	/**
     * Read Telium full serial number
     *
     * @param serialNumber  Full Serial number of the remote device.
     * @return        true if the method succeeded.
     *                false if the method failed.
     */
	boolean getFullSerialNumber(out byte[] serialNumber);
	
	/**
     * Print bitmap on remote device.
     *
     * @param bmp     Bitmap to be printed
     * @param result  Result of the operation (0=OK, 1=KO)
     * @return        true if the method succeeded (see result for command result).
     *                false if the method failed.
     */
    boolean printBitmapObject(in Bitmap bmp, out byte[] result);
    
    /**
     * Add TCP/IP bridge between Android device and Telium device.
     * In case of bridge from Android to Telium, listen is done only on localhost. 
     *
     * @param port		TCP port to connect to or to listen (depending on direction parameter)
     * @param direction Bridge direction: 0 means Android connects to Telium TCP port
     *                                    1 means Telium connects to Android TCP port
     */
    void addDynamicBridgeLocal(in int port, in int direction);
}