/*
 * clsLocation.java
 *
 * Created on 10 June 2005, 10:05
 *
 * @author lambertonr
 *
 * Copyright (c) Fujitsu Services 2005.
 *
 * Change history
 * When             Who             What
 * 09.01.2006       abjm            Serialize UID Removed.
 *                                  Commented Out code removed.
 * 02.06.2006       RS              Add THOLocation.
 * 30.11.2006       MAC             Amended to produce javadoc.
 * 14.06.2007       abjm            IR12184: equals() method signature changed
 *                                  from overload to override.
 * 28.10.2008       Dave Oakley     Changed deriveCountryCode() to use logical And
 *                                  operations instead of bitwise And operations
 *                                  to reduce any potential unnecessary expression
 *                                  processing.
 * 10.01.2013       Rob Swain       RAIL-250: amended getPTEZones() to check for
 *                                  empty string as well as null in test.
 *                                  Added zoneInd attribute and use this in
 *                                  getPTEZones() instead of zoneCode.
 */


package com.example.twolibs.LocationBL;
//JP_EDIT package JourneyPlanner.IJQ.BusinessClasses;

import java.io.Serializable;
import java.util.Date;
import java.util.Vector;
//JP_EDIT import static JourneyPlanner.IJQ.CONSTANTS_IJQI.LLNKTYPE_FARESGROUPMEMBER;
//JP_EDIT import static JourneyPlanner.IJQ.CONSTANTS_IJQI.LLNKTYPE_TIMETABLEGROUPMEMBER;
//JP_EDIT import static JourneyPlanner.IJQ.CONSTANTS_IJQI.LNKQUAL1_NOQUALIFIER;


/** This is a "light" location object used in the Enquiry Manager,
 *  and related subsystems.
 *  It does not include the station info or any other attributes that
 *  are not used in general enquiries, for example it only requires attributes
 *  required for fare calculation, timetable enquiries, ticket handoff etc.
 *  As this specification is developed, further attributes may be added,
 *  as required (eg for ENRG processing). */
public class clsLocation implements Cloneable,Serializable {

    private static final long serialVersionUID = 1352874795293552470L;

    /** The unique 7 digit code identifying the location */
    public String UIC7;

    /** NLC code of the location */
    public String NLCCode;

    /** Links to locations associated with this location */
    public Vector<clsLocationLink> locationLinks;

    /** Special facilities that exist at this location */
    public String specialFacilities;

    /** A list of railcards that are valid at this location */
    public Vector<String> railcards;

    /** CRS code of location, or NIL if location has no CRS code */
    public String CRSCode;

    /** Group code of the location */
    public String groupCode;

    /** County code of the location */
    public String countyCode;

    /** Zone NLC code of the location */
    public String zoneCode;

    /** Zone indicator for the location for PTE (Public Transport Executive) locations. */
    public String zoneInd;

    /** PTE code of the location */
    public String PTECode;

    /** The first date on which the record is valid */
    public Date   startDate;

    /** The last date on which the record is valid */
    public Date   endDate;

    /** Location description */
    public String description;

    /** Indicates whether the location is an ENRG station */
    public boolean isAnENRGStation;

    /** Fare zones in which the location occurs */
    public Vector <clsZone> fareZones;

    /** Zones in which the location occurs */
    public Vector <clsZone> PTEZones;

    /** Ticket Handoff Location info */
    public clsIJQLocation THOLocation;

    /** Boundary zone number, present when a location is on a zonal boundary.
     * Used with ZoneInd to identify the two zones in which the location occurs. */
    public String boundaryZoneNo;

    /** Station hierarchy level. */
    public String hierarchy;

    /** Creates a new instance of clsLocation */
    public clsLocation() {
        // create any object attributes that must not be nil
        this.endDate = new Date();
        this.startDate = new Date();
        this.locationLinks = new Vector<clsLocationLink>();
        this.railcards = new Vector<String>();
        this.fareZones = new Vector<clsZone>();
        this.PTEZones = new Vector<clsZone>();
    }


    /**
     * Returns TRUE if GroupCode not NIL and GroupCode NOT = NLCCode
     *
     * @return  true or false depending upon whether GroupCode not NIL and GroupCode NOT = NLCCode
     */
    public boolean isMemberOfAFaresGroup() {
        if ((groupCode!=null) && (groupCode.equals(NLCCode)==false)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns TRUE if LocationLinks contains any links
     *  with Type LINKTYPE_FARESGROUPMEMBER
     *
     * @return  true or false depending upon whether LocationLinks contains any links
     *          with Type LINKTYPE_FARESGROUPMEMBER.
     */
    public boolean isAFaresGroup() {
        boolean ret = false;

        if (locationLinks!=null) {

            for (clsLocationLink locLink : locationLinks) {
                if (locLink.type==CONSTANTS_IJQI.LLNKTYPE_FARESGROUPMEMBER) {  // JP_EDIT
                    ret = true;
                    break;
                }
            }
        }

        return ret;
    }

    /**
     * Returns TRUE if LocationLinks contains any links
     *  with Type LINKTYPE_TIMETABLEGROUPMEMBER
     *
     * @return   true or false depending upon whether LocationLinks contains any links
     *          with Type LINKTYPE_TIMETABLEGROUPMEMBER.
     */
    public boolean isATTGroup() {
        boolean ret = false;

        if (locationLinks!=null) {
            for (clsLocationLink locLink : locationLinks) {
                if (locLink.type==CONSTANTS_IJQI.LLNKTYPE_TIMETABLEGROUPMEMBER) { // JP_EDIT
                    ret = true;
                    break;
                }
            }
        }

        return ret;
    }

    /**
     * Returns CountryCode = S, E or Space depending on the
     *  value of the CountyCode attribute
     *
     * @return  countryCode string value of S, E or space.
     */
    public String deriveCountryCode() {
        String countryCode = " ";       // Default to a space

        try {
            int countyNumber = Integer.valueOf(countyCode).intValue();
            if ((countyNumber>=38) && (countyNumber<=43)) {
                countryCode = "S";
            } else {
                countryCode = "E";
            }

        } catch (NumberFormatException ex) {
            // Not numeric
            countryCode = " ";  // space
        }

        return countryCode;
    }

    /**
     * Returns all the location links with Type = Specified type type and
     *  with the qualifier = Specified qualifier type a(if qualifier supplied).
     *  If specified qualifier = LNKQUAL1_NOQUALIFIER (-1) then the method
     *  returns all the location links with Type = Specified type irrespective
     *  of its qualifier.
     *
     * @param type
     * @param qualifier
     * @return  locLinks
     */
    public Vector<clsLocationLink>
            getLocationLinksForType( int type, int qualifier ) {

        Vector<clsLocationLink> locLinks = new Vector<clsLocationLink>();

        if (locationLinks!=null) {
            for (clsLocationLink locLink : locationLinks) {
                if (locLink.type==type) {
                    if ( qualifier == CONSTANTS_IJQI.LNKQUAL1_NOQUALIFIER ) {// JP_EDIT
                        locLinks.add(locLink);
                    } else {
                        if ( locLink.qualifier == qualifier  ) {
                            locLinks.add(locLink);
                        }
                    }
                }
            }
        }

        return locLinks;
    }

    /**
     * Returns TRUE if the supplied RailcardCode is included
     *  in the vector Railcards
     * @param   railcardCode
     * @return  true or false depending on whether the supplied RailcardCode is
     *          included in the vector Railcards.
     */
    public boolean hasRailcard(String railcardCode) {
        if (this.railcards!=null) {
            for (String eachRailcard : this.railcards) {
                if  (eachRailcard.equals(railcardCode)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Returns Zones with zoneCode included in it, when zoneCode is not null.
     * @return  Zones in which the location occurs.
     */
    public clsPTEZones getPTEZones() {
        if ((this.zoneInd == null) || (this.zoneInd.isEmpty())) {
            return null;
        } else {
            clsPTEZones PTEZones = new clsPTEZones();
            PTEZones.zones = this.zoneInd;
            if (this.boundaryZoneNo != null) {
                PTEZones.zones = PTEZones.zones + this.boundaryZoneNo;
                PTEZones.boundaryLoc = true;
            }

            return PTEZones;
        }
    }

    /** Returns a string representation of this object, for logging purposes */
    public String toString() {
        StringBuffer sData = new StringBuffer();
        final String PREFIX = "\n                    ";
        sData.append("\n      clsLocation:");
        sData.append(PREFIX + "UIC7=" + this.UIC7 +
                "\t NLCCode=" + this.NLCCode +
                "\t CRSCode=" + this.CRSCode +
                "\t groupCode=" + this.groupCode +
                "\t countyCode=" + this.countyCode +
                "\t zoneCode=" + this.zoneCode +
                "\t zoneInd=" + this.zoneInd +
                "\t PTECode=" + this.PTECode +
                "\t isAnENRGStation=" + this.isAnENRGStation +
                PREFIX + "description=" + this.description +
                "\t startDate=" + this.startDate +
                "\t endDate=" + this.endDate +
                PREFIX + "locationLinks=" + this.locationLinks +
                PREFIX + "specialFacilities=" + this.specialFacilities +
                PREFIX + "railcards=" + this.railcards +
                PREFIX + "fareZones=" + this.fareZones +
                PREFIX + "PTEZones=" + this.PTEZones +
                PREFIX + "THOLocation=" + this.THOLocation +
                PREFIX + "boundaryZoneNo=" + this.boundaryZoneNo +
                PREFIX + "hierarchy=" + this.hierarchy);

        return sData.toString();
    }

    /**
     * Overridden method, which compares specific attributes of this
     *  object with another, instead of the default method which
     *  just compares the reference.
     * This version compares the following attributes:
     *      CRSCode, UIC7, NLCCode
     *
     * @param locObj    location to be compared
     * @return          true if objects match
     */
    public boolean equals(Object locObj) {

        if  (locObj instanceof clsLocation) {
            clsLocation loc = (clsLocation) locObj;
            if ((loc!=null) &&
                (loc.CRSCode.equals(this.CRSCode)) &&
                (loc.UIC7.equals(this.UIC7)) &&
                (loc.NLCCode.equals(this.NLCCode))) {
            return true;
            }
        }
        return false;
    }

}
