/*
 * clsUtilities.java
 *
 * Created on 18 October 2005, 10:50
 *
 * @author OakleyDJ
 *
 * Copyright (c) Fujitsu Services 2005.
 *
 * Change history
 * When             Who             What
 * 18/10/2005       Dave Oakley     New method for useful reusable code.
 * 05/01/2006       abjm            trimLeading and trimTrailing methods
 *                                  introduced.
 *                                  padWithSpaces method introduced.
 * 11.01.2006       abjm            formatHashDate method removed.
 * 05.04.2007       Dave Oakley     Added stringsAreDiff() method.
 * 24.10.2007       R.Lamberton     Moved the 4 methods from SessionManager, that
 *                                  convert strings to Title/Name case, so that they
 *                                  can be reused elsewhere.
 * 02.06.2008       R.Lamberton     Moved getJPHomeDir from out of the GUI
 *                                  project so it can be reused.
 * 15.08.2008       D.Oakley        Renamed getJPHomeDir to getHomeDir() and added
 *                                  code to determine whether we are using a journey
 *                                  planner or data maintenance application.
 * 23.10.2008       Dave Oakley     Added static calendar to reduce the performance
 *                                  impact of getInstance() calls.
 * 27.10.2008       Dave Oakley     setTimeZero() now returns null if null date supplied.
 */

package com.example.twolibs.TimeTableBL;
//JP_EDIT package JourneyPlanner.Utilities;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.util.Calendar;
import java.util.Date;
import java.util.Vector;

/**
 * This class is holds useful reusable components, such as object cloners.
 *
 */
public class clsUtilities {
   
    private static Calendar convCalendar = null;
    
    /** Creates a new instance of clsUtilities.  This method cannot be 
     *  instantiated as it will only contain static methods.
     */
    private clsUtilities() {
    }
    /**
     * The <code>stringsAreDiff</code> method returns true if the two strings
     * are different ignoring case.  If both strings are null or the same 
     * (ignoring case) then false is returned otherwise true is returned
     * @param   string1     The first <code>string</code> to compare
     * @param   string2     The second <code>string</code> to compare
     * @return              true if the strings are different, false if they are 
     *                      the same ignoring case.
     *                  
     */
    public static boolean stringsAreDiff(String string1, String string2) {
        if (((string1 == null) && (string2 != null)) ||
                ((string1 != null) && (string2 == null))) {
            return true;
        }
        if (((string1 != null) && (string2 != null)) &&
                (string1.compareToIgnoreCase(string2)!=0)) {
            return true;
        }
        return false;
    }
     
    /**
     * The <code>deepClone</code> method returns a deep copy of the supplied object.
     * The user is expected to cast the returned object to the type supplied.
     * The object supplied must extend the java.io.Serializable interface.
     *
     * @param   toClone The <code>Object</code> to clone.
     * @return          A deep copy of <code>toClone</code> is returned.
     */
     
    static public Object deepClone(Object toClone ) {
        if ( toClone == null )
        {
            return null;
        }
        
        Object clonedObject = null;
        
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream( baos );
            oos.writeObject( toClone );
            oos.close();
            
            // Read the object back into another object.
            ByteArrayInputStream bais = new ByteArrayInputStream( baos.toByteArray() );
            ObjectInputStream ois = new ObjectInputStream( bais );
            clonedObject = ois.readObject();          
            ois.close();
            
        }
        catch ( Exception e)
        {
            // The following exceptions may be thrown, there is no specific
            // distinction we need to make between any failure, hence the generic
            // catch used.
            //
            // ClassNotFoundException - Class of a serialized object cannot be found. 
            // InvalidClassException - Something is wrong with a class used by serialization. 
            // StreamCorruptedException - Control information in the stream is inconsistent. 
            // OptionalDataException - Primitive data was found in the stream instead of objects. 
            // IOException - Any of the usual Input/Output related exceptions.
            e.printStackTrace();
        }
        return clonedObject;
    }

    /** 
     * Returns the supplied string having first appended
     * trailing spaces so that its length equals the
     * supplied textLength.
     *
     * @param textValue The String to be reformatted with trailing spaces.
     * @param textLength The length of the String to be returned.
     * @return   The supplied String text without leading spaces.
     */
    static public String padWithSpaces(String textValue,
                                       int    textLength) {
        StringBuffer textBuffer;

        if  (textValue == null)	{
            return null;
        }

        textBuffer = new StringBuffer(textValue);

        while (textBuffer.length() < textLength) {
            textBuffer.append(" ");
        }
    
        return textBuffer.toString();  
    
    }
    
    /**
     * The <code>setTimeZero</code> method returns a new Date object with the
     * time element removed and set to 0.
     *
     * @param   sourceDateTime The <code>Date</code> to truncate to midnight.
     * @return          A new instance of <code>sourceDateTime</code> to be 
     *                  returned with the time component removed.
     */
    synchronized public static Date setTimeZero( Date sourceDateTime  )
    {
        if (sourceDateTime == null) {
            return null;
        }
        
        Date truncatedDateTime = new Date();
        if (convCalendar == null) {
            convCalendar = Calendar.getInstance();
        }
        convCalendar.setTime( sourceDateTime );
        convCalendar.set( Calendar.HOUR_OF_DAY, 0 );
        convCalendar.set( Calendar.MINUTE, 0 );
        convCalendar.set( Calendar.SECOND, 0 );
        convCalendar.set( Calendar.MILLISECOND, 0 );
        truncatedDateTime.setTime( convCalendar.getTimeInMillis());
        return truncatedDateTime;
    }

    /** 
     * Returns the supplied string having first removed
     * all leading whitespace characters.
     *
     * @param in The String to be reformatted without leading spaces.
     * @return   The supplied String text without leading spaces.
     */
    static public String trimLeading(String in) {

        if  (in == null) {
            return null;
        }
        
        int x=(in.length());
        for (int j=0; j<x; j++) {
            if ( in.charAt(j)==' ' ) {
                continue;
            } else {
                return in.substring(j,x);
            }
        }

        return "";

    }

    /** 
     * Returns the supplied string having first removed
     * all trailing whitespace characters.
     *
     * @param in The String to be reformatted without trailing spaces.
     * @return   The supplied String text without trailing spaces.
     */
    static public String trimTrailing(String in) {

        if  (in == null) {
            return null;
        }
        
        int x=(in.length() - 1);
        for (int j=x; j>=0; j--) {
            if ( in.charAt(j)==' ' ) {
                continue;
            } else {
                return in.substring(0,j+1);
            }
        }

        return "";

    }    

    /**
     * This method is supplied with a string of characters which it converts to
     * title case i.e. each of the separate words in the string is converted to
     * lower case and its first letter capitalised.
     * The translation has exceptions for certain words and symbols.
     *
     * @param  textUpper            Upper case location description.
     * @return String               Location description after conversion
     */
    static public String textToName(String textUpper) {
        String textResult = "";
        String textFinal;
        
        // If empty string supplied then return an empty string
        if (textUpper.length() < 1) {
            return "";
        }
        
        String textLower = textUpper.toLowerCase();
        
        char[] contents = textLower.toCharArray();
        char lastChar = ' ';
        
        for (int i = 0; i < contents.length; i++) {
            if ((lastChar        == ' ')  &&
                    (contents[i]     == 'o')  &&
                    (i + 2 < contents.length) &&
                    (contents[i + 1] == 'n')  &&
                    (contents[i + 2] == ' ')) {
                // Found an 'on' so append this character (o) to the results as is.
                textResult = textResult.concat(new String(new char[]{contents[i]}));
            } else if ((lastChar == ' ') ||
                    (lastChar == '&') ||
                    (lastChar == '-') ||
                    (lastChar == '+') ||
                    (lastChar == '(')) {
                // Append this character to the results in upper case
                textResult = textResult.concat(new String(new char[]{contents[i]}).toUpperCase());
            } else {
                // Append this character to the results as is.
                textResult = textResult.concat(new String(new char[]{contents[i]}));
            }
            lastChar = contents[i];
        }
        
        // Look for any discrete "Br"s and convert to "BR"
        textFinal = makeWordUpperCase(textResult, "Br");
        
        textResult = textFinal;
        
        // Look for any discrete "Ni"s and convert to "NI"
        textFinal = makeWordUpperCase(textResult, "Ni");
        
        return textFinal;
    }
    
    
    /**
     * This method looks for any discrete ocurrences of a selected word and
     * converts them to upper case, returning the converted string.
     *
     * @param  sourceString         String to be searched for the specified word
     * @param  wordToFind           Word to be searched for in the source string
     * @return String               String after conversion
     */
    static public String makeWordUpperCase(String sourceString, String wordToFind) {
        
        int stringLocation;
        String textFinal = new String(sourceString);
        
        char[] contents = sourceString.toCharArray();
        
        stringLocation = sourceString.indexOf(wordToFind);
        while (stringLocation > -1) {
            
            if ((stringLocation == 0) &&
                    ((sourceString.length() == wordToFind.length()) ||
                    (contents[stringLocation + wordToFind.length()] == ' '))) {
                // Replace with word at the start (or the whole) of the line
                textFinal = wordToFind.toUpperCase();
                if (sourceString.length() > wordToFind.length()) {
                    textFinal = textFinal.concat(sourceString.substring(stringLocation + wordToFind.length()));
                }
                
            } else if (stringLocation == sourceString.length() - wordToFind.length()) {
                if (contents[stringLocation -1] == ' ') {
                    // Replace with upper case wordToFind at the end of the name
                    textFinal = textFinal.substring(0, stringLocation);
                    textFinal = textFinal.concat(wordToFind.toUpperCase());
                }
                
            } else if ((contents[stringLocation + wordToFind.length()] == ' ') &&
                    (contents[stringLocation - 1] == ' ')) {
                // Replace with upper case wordToFind in the middle of the name
                textFinal = textFinal.substring(0, stringLocation);
                textFinal = textFinal.concat(wordToFind.toUpperCase());
                textFinal = textFinal.concat(sourceString.substring(stringLocation + wordToFind.length()));
            }
            
            // Search again
            stringLocation = sourceString.indexOf(wordToFind, stringLocation + 1);
        }
        
        return textFinal;
    }
    
    
    /**
     * This method is supplied with a string of characters which it converts to
     * lower case except for the first letter.
     *
     * @param  textUpper            Upper case item description.
     * @return String               Item description after conversion
     */
    static public String textToDescription(String textUpper) {
        
        // If empty string supplied then return an empty string
        if (textUpper.length() < 1) {
            return "";
        }
        
        String textLower  = textUpper.toLowerCase();
        String textResult = textUpper.substring(0,1).toUpperCase().concat(
                textLower.substring(1));
        
        return textResult;
    }
    
    
    /**
     * This method is supplied with a string of characters which it converts to
     * lower case except for the first letter of each word.
     *
     * @param  textUpper            Upper case item description.
     * @return String               Item description after conversion
     */
    static public String textToTitleCase(String textUpper) {
        
        // If empty string supplied then return an empty string
        if (textUpper.length() < 1) {
            return "";
        }
        
        String textLower = textUpper.toLowerCase();
        
        String textResult = "";
        char[] contents = textLower.toCharArray();
        char lastChar = ' ';
        
        for (int i = 0; i < contents.length; i++) {
            if (lastChar == ' ') {
                // Append this character to the results in upper case
                textResult = textResult.concat(new String(new char[]{contents[i]}).toUpperCase());
            } else {
                // Append this character to the results as is.
                textResult = textResult.concat(new String(new char[]{contents[i]}));
            }
            lastChar = contents[i];
        }
        
        return textResult;
    }
    
    /**
     *  This method returns the directory to be used as the stem for journey 
     *  planner related files, e.g. client cache, handoff files.
     *  @return The absolute pathname for the String to be used for file creation.
     */
    @SuppressWarnings("unchecked")
    static public String getHomeDir() {
        
        String JPHomeString = null;
        File   file         = null;
//        String filename     = null;

        // Determine if we are JourneyPlanning application or a data maintenance
        // application.
        String applicationName = "JourneyPlanner";
        try {
            @SuppressWarnings("unused")
            final Class aClass = Class.forName("DataMaintenance.GUI.MainMenu.MainMenu");                    

            // We must be using the data maintenance application.
            applicationName = "DataMaintenance";
        } catch (ClassNotFoundException exception) {                    
        }

        // Locate any existing cache which has been stored locally
        String homeDirectory = System.getProperty("user.home");
        if ((homeDirectory.toLowerCase().endsWith("windows")) ||
            (homeDirectory.toLowerCase().endsWith("winnt"))) {
            homeDirectory = System.getProperty("java.io.tmpdir");
        }

        // Create a directory list with specific directory names
        Vector<String> directoryNames = new Vector<String>();
        if ((homeDirectory != null) && (homeDirectory.length() !=0 )) {
            directoryNames.add(homeDirectory);
        }
        directoryNames.add("C:\\Temp");
        directoryNames.add("C:\\Windows\\Temp");
        directoryNames.add("C:\\WINNT\\Temp");
        directoryNames.add("D:\\Temp");
        directoryNames.add("D:\\Windows\\Temp");
        directoryNames.add("D:\\WINNT\\Temp");
        directoryNames.add("C:\\");
        directoryNames.add("D:\\");

        // Loop round the directory list and check if any of the directories
        // exist and are writeable (grab the first one that does from this list).
        for (String latestDirectory: directoryNames) {
            
            // Check that file can be written to. If not try the next one
            file = new File(latestDirectory);
            if (!file.exists()) {
                continue;
            }
            else if (!file.canWrite()) {
                continue;
            }
            else {
 
                    
                // We have a writeable base directory.
                JPHomeString = latestDirectory + "\\" + applicationName;
                File JPHomeFile = new File(JPHomeString);
                
                if (JPHomeFile.exists() && JPHomeFile.canWrite()) {
                    // We have already created the journey planners home directory
                    // at some point so use this.
                    return JPHomeString;
                } else {
                    // First time in for this directory, so create it and this.
                    if (JPHomeFile.mkdir()) {
                        return JPHomeString;
                    }
                    // We have failed to create the directory so loop around and
                    // look for another one.
                }
            }
        }
        // We have not yet returned with a valid writeable directory so return
        // null to indicate an error.
        return null;                
    }        
}
