package com.robotsandpencils.biotest

import android.annotation.SuppressLint
import android.os.Bundle
import android.os.Looper
import android.view.View
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import com.robotsandpencils.biotest.data.pref.PreferenceManager
import com.robotsandpencils.biotest.data.transformer.Async
import com.robotsandpencils.biotest.domain.error.DomainError
import com.robotsandpencils.biotest.domain.repository.AuthResponse
import com.robotsandpencils.biotest.domain.repository.AuthTokenRepository
import com.robotsandpencils.biotest.domain.repository.CurrentUser
import com.robotsandpencils.biotest.domain.repository.SessionRepository
import com.robotsandpencils.biotest.domain.usecase.execute
import com.robotsandpencils.biotest.domain.usecase.login.RefreshTokenLoginUseCase
import com.robotsandpencils.biotest.system.repository.BiometricsRepository
import com.robotsandpencils.biotest.system.usecase.biometrics.AreBionmetricsCredentialsStoredUseCase
import com.robotsandpencils.biotest.system.usecase.biometrics.BiometricsLoginUseCase
import com.robotsandpencils.biotest.system.usecase.biometrics.ClearBiometricsCredentialsUseCase
import com.robotsandpencils.biotest.system.usecase.biometrics.EnrollBiometricsUseCase
import io.reactivex.Completable
import io.reactivex.Observable
import io.reactivex.Single
import io.reactivex.android.plugins.RxAndroidPlugins.setInitMainThreadSchedulerHandler
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.plugins.RxJavaPlugins
import io.reactivex.rxkotlin.subscribeBy
import io.reactivex.subjects.BehaviorSubject
import kotlinx.android.synthetic.main.activity_main.*
import timber.log.Timber
import java.util.*

class MainActivity : AppCompatActivity() {

    private lateinit var areBiometricsCredentialsStoredUseCase: AreBionmetricsCredentialsStoredUseCase
    private lateinit var enrollBiometricsUseCase: EnrollBiometricsUseCase
    private lateinit var biometricsLoginUseCase: BiometricsLoginUseCase
    private lateinit var clearBiometricsCredentialsUseCase: ClearBiometricsCredentialsUseCase

    private lateinit var biometricsEnabled: Observable<Boolean>

    private val disposables = CompositeDisposable()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)

        // Didn't want to implement the entire Dagger tree, so set up mock dependencies for the use
        // cases to work.
        initializeDependencies()

        buttonEnroll.setOnClickListener {

            val request = EnrollBiometricsUseCase.EnrollRequest(
                this@MainActivity, "user@test.com"
            )

            enrollBiometricsUseCase.execute(request)
                .subscribeBy(
                    onError = { err ->
                        Toast.makeText(
                            this@MainActivity,
                            err.localizedMessage,
                            Toast.LENGTH_LONG
                        ).show()
                    },
                    onComplete = {
                    }
                ).also { disposables.add(it) }
        }

        buttonClear.setOnClickListener {
            clearBiometricsCredentialsUseCase.execute()
                .subscribeBy(
                    onError = { err ->
                        Toast.makeText(
                            this@MainActivity,
                            err.localizedMessage,
                            Toast.LENGTH_LONG
                        ).show()
                    },
                    onComplete = {
                    }
                )
                .also {
                    disposables.add(it)
                }
        }

        buttonLogin.setOnClickListener {
            performLogin()
        }

        biometricsEnabled.subscribeBy(
            onError = {},
            onNext = {
                renderButtons(it)
            }
        ).also {
            disposables.add(it)
        }
    }

    private fun renderButtons(isEnabled: Boolean) {
        when (isEnabled) {
            true -> {
                buttonEnroll.visibility = View.GONE
                buttonClear.visibility = View.VISIBLE
                buttonLogin.visibility = View.VISIBLE
            }
            false -> {
                buttonEnroll.visibility = View.VISIBLE
                buttonClear.visibility = View.GONE
                buttonLogin.visibility = View.GONE
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        disposables.clear()
    }

    override fun onResumeFragments() {
        super.onResumeFragments()

        areBiometricsCredentialsStoredUseCase.execute(Async)
            .subscribeBy(onError = {},
                onSuccess = {
                    if (it) {
                        performLogin()
                    }
                }).also { disposables.add(it) }
    }

    private fun performLogin() {
        biometricsLoginUseCase.execute(this)
            .subscribeBy(
                onError = { err ->
                    Toast.makeText(
                        this@MainActivity,
                        err.localizedMessage,
                        Toast.LENGTH_LONG
                    ).show()
                },
                onComplete = {
                    Toast.makeText(
                        this@MainActivity,
                        "Authenticated",
                        Toast.LENGTH_LONG
                    ).show()
                }
            ).also {
                disposables.add(it)
            }
    }

    private fun initializeDependencies() {

        setInitMainThreadSchedulerHandler {
            AndroidSchedulers.from(Looper.getMainLooper(), true)
        }

        RxJavaPlugins.setErrorHandler {
            Timber.e(it, "UNHANDLED EXCEPTION - Continuing")
        }

        val authTokenRepository = object : AuthTokenRepository {

            val refreshTokenSubject = BehaviorSubject.create<String>()

            override var refreshToken: String
                get() = "refreshtoken"
                set(value) {}
            override var accessToken: String
                get() = TODO("authTokenRepository:accessToken not implemented") //To change initializer of created properties use File | Settings | File Templates.
                set(value) {}
            override val userId: String
                get() = TODO("authTokenRepository:userId not implemented") //To change initializer of created properties use File | Settings | File Templates.
            override val password: String
                get() = TODO("authTokenRepository:password not implemented") //To change initializer of created properties use File | Settings | File Templates.

            override fun fireRefreshTokenUpdate(refreshToken: String) {
                refreshTokenSubject.onNext(refreshToken)
            }

            override fun refreshTokenUpdated(): Observable<String> {
                return refreshTokenSubject
            }

            override fun hasAuthToken(): Boolean {
                return true
            }
        }

        val sessionRepository = object : SessionRepository {
            override fun logIn(refreshToken: String): Single<AuthResponse> {
                return Single.just(
                    AuthResponse(
                        access_token = "accessToken",
                        token_type = "Bearer",
                        refresh_token = "refreshToken2",
                        expires_in = "100",
                        scope = "sesson",
                        jti = "jti",
                        varo_client_id = UUID.randomUUID()
                    )
                )
            }

            override fun logOut(revokeAndRemoveTokens: Boolean): Completable {
                TODO("sessionRepository:logOut not implemented") //To change body of created functions use File | Settings | File Templates.
            }

            override fun currentUser(): Observable<CurrentUser> {
                TODO("sessionRepository:currentUser not implemented") //To change body of created functions use File | Settings | File Templates.
            }

            override val isLoggedIn: Boolean
                get() = TODO("sessionRepository:isLoggedIn not implemented") //To change initializer of created properties use File | Settings | File Templates.

            override fun backgroundSessionTimeout(): Observable<Boolean> {
                TODO("sessionRepository:backgroundSessionTimeout not implemented") //To change body of created functions use File | Settings | File Templates.
            }

            override fun fireBackgroundSessionTimeout() {
                TODO("sessionRepository:fireBackgroundSessionTimeout not implemented") //To change body of created functions use File | Settings | File Templates.
            }
        }

        DomainError.errorResolver = object : DomainError.Companion.DomainErrorResolver {
            override fun localizedMessage(error: DomainError): String {
                return error.name
            }

        }

        val refreshTokenLoginUseCase = RefreshTokenLoginUseCase(
            sessionRepository = sessionRepository,
            authTokenRepository = authTokenRepository
        )

        val preferenceManager = PreferenceManager(context = this, appId = "biometrics")

        biometricsEnabled = preferenceManager.biometricsEnabled.asObservable()

        @SuppressLint("NewApi")
        val biometricsRepository = BiometricsRepository(
            authTokenRepository = authTokenRepository,
            preferenceManager = preferenceManager,
            refreshTokenLoginUseCase = refreshTokenLoginUseCase
        )

        areBiometricsCredentialsStoredUseCase = AreBionmetricsCredentialsStoredUseCase(
            biometricsRepository = biometricsRepository
        )

        enrollBiometricsUseCase = EnrollBiometricsUseCase(
            biometricsRepository = biometricsRepository,
            authTokenRepository = authTokenRepository
        )

        biometricsLoginUseCase = BiometricsLoginUseCase(
            biometricsRepository = biometricsRepository
        )

        clearBiometricsCredentialsUseCase = ClearBiometricsCredentialsUseCase(
            biometricsRepository = biometricsRepository
        )
    }
}
