package com.robotsandpencils.biotest.data.pref

import android.annotation.SuppressLint
import android.content.Context
import android.content.SharedPreferences
import androidx.annotation.Keep
import com.f2prateek.rx.preferences2.Preference
import com.f2prateek.rx.preferences2.RxSharedPreferences

@Keep
class PreferenceManager(context: Context, appId: String) {

    private val preferences: SharedPreferences =
        context.getSharedPreferences("$appId.Preferences", Context.MODE_PRIVATE)

    private val rxPreferences = RxSharedPreferences.create(preferences)

    @SuppressLint("CommitPrefEdits", "ApplySharedPref")
    fun putBoolean(key: String, value: Boolean) {
        preferences.edit().putBoolean(key, value).commit()
    }

    fun getBoolean(key: String, defaultValue: Boolean = false): Boolean =
        preferences.getBoolean(key, defaultValue)

    @SuppressLint("CommitPrefEdits", "ApplySharedPref")
    fun putString(key: String, value: String? = null) {
        if (value != null) {
            preferences.edit().putString(key, value).commit()
        } else {
            preferences.edit().remove(key).commit()
        }
    }

    @SuppressLint("ApplySharedPref")
    fun putFloat(key: String, value: Float) {
        preferences.edit().putFloat(key, value).commit()
    }

    fun getString(key: String): String? = preferences.getString(key, null)

    fun getString(key: String, defaultValue: String): String = getString(key) ?: defaultValue

    @SuppressLint("ApplySharedPref")
    fun remove(key: String) = preferences.edit().remove(key).commit()

    fun contains(key: String): Boolean {
        return preferences.contains(key)
    }

    fun getBooleanPreference(key: String, defaultValue: Boolean = false): Preference<Boolean> {
        return rxPreferences.getBoolean(key, defaultValue)
    }

    fun getStringPreference(key: String, defaultValue: String = ""): Preference<String> {
        return rxPreferences.getString(key, defaultValue)
    }

    fun getLongPreference(key: String, defaultValue: Long = -1): Preference<Long> {
        return rxPreferences.getLong(key, defaultValue)
    }

    fun getFloatPreference(key: String, defaultValue: Float = 0.0f): Preference<Float> {
        return rxPreferences.getFloat(key, defaultValue)
    }

    val demoModeEnabled = getBooleanPreference(DEMO_MODE_ENABLED, false)
    val biometricsEnabled = getBooleanPreference(BIOMETRICS_ENABLED, false)

    val versionCode = getLongPreference(VERSION_CODE, -1)
    val versionName = getStringPreference(VERSION_NAME, "")

    val launcherEmail = getStringPreference(LAUNCHER_EMAIL, "")

    val directDepositDismissed = getBooleanPreference(DIRECT_DEPOSIT_DISMISSED, defaultValue = false)

    val suggestFeatureTermsAccepted = getBooleanPreference(SUGGEST_FEATURE_TERMS_ACCEPTED, false)

    val backgroundedTimestamp = getLongPreference(key = BACKGROUNDED_TIMESTAMP, defaultValue = UNASSIGNED_DATE)

    companion object {
        // Preference Keys
        // Since the value of the key doesn't matter, we might as well use UUIDs
        const val DEMO_MODE_ENABLED = "c8e04c45-4af9-410e-9e35-7c5e24f4b874"
        const val BIOMETRICS_ENABLED = "fe589bce-8141-11e8-adc0-fa7ae01bbebc"
        const val DEVICE_UUID = "F22BCBAE-741C-42F2-BBA0-823F8AE66A78"

        const val VERSION_CODE = "54dce9aa-8c1e-11e8-9eb6-529269fb1459"
        const val VERSION_NAME = "5e1825a2-8c1e-11e8-9eb6-529269fb1459"

        const val LAUNCHER_EMAIL = "29cd62ec-8c25-11e8-9eb6-529269fb1459"

        const val INITIAL_MAP_LATITUDE = "d01c7116-8769-4823-a631-3c78d206a4a2"
        const val INITIAL_MAP_LONGITUDE = "b3f9ffaf-a4cd-41e0-aa4b-fcdf03171642"
        const val INITIAL_MAP_ZOOM = "f5e2a1d0-659e-4417-b8ff-72ed89bb013c"

        const val DIRECT_DEPOSIT_DISMISSED = "b5690b81-4b88-480b-b9a2-62df0a8e09d0"

        const val SUGGEST_FEATURE_TERMS_ACCEPTED = "d2b8c05e-bf88-11e8-a355-529269fb1459"

        const val BACKGROUNDED_TIMESTAMP = "4f303bb6-b86d-4d87-81b2-16b2f70a4186"

        const val ATM_LOCATOR_USED = "39bb1450-55f5-4b90-8971-c8db6bcd5905"

        const val UNASSIGNED_DATE = -1L
    }
}
