package com.robotsandpencils.biotest.domain.usecase.login

import com.robotsandpencils.biotest.domain.repository.AuthTokenRepository
import com.robotsandpencils.biotest.domain.repository.SessionRepository
import com.robotsandpencils.biotest.domain.usecase.CompletableUseCase
import io.reactivex.Completable

/**
 * Login with the refresh token. For use during a biometric login.
 * Created by aoedlund on 3/26/18.
 */
class RefreshTokenLoginUseCase(
    private val sessionRepository: SessionRepository,
    private val authTokenRepository: AuthTokenRepository
) : CompletableUseCase<String>() {

    public override fun createCompletable(data: String): Completable {
        /*
         * Reset the access tokens before calling the auth endpoint to make the HTTP client use
         * Basic authentication with the built in client secret & key.
         *
         * This is needed to reset things if one of the calls after the auth calls fails, so
         * the user is stuck on the login screen. (For example if the /me endpoint times out,
         * which happens from time to time in the dev environments)
         */
        authTokenRepository.accessToken = ""
        authTokenRepository.refreshToken = ""

        return sessionRepository.logIn(data)
            .doOnSuccess {
                authTokenRepository.accessToken = it.access_token
                authTokenRepository.refreshToken = it.refresh_token
            }
            .doOnError {
                // If any of the above calls fail, reset the access tokens
                authTokenRepository.accessToken = ""
                authTokenRepository.refreshToken = ""
            }
            .ignoreElement()
    }
}
