package com.robotsandpencils.biotest.system.repository

import android.app.Activity
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyPermanentlyInvalidatedException
import android.security.keystore.KeyProperties
import androidx.annotation.RequiresApi
import androidx.biometric.BiometricPrompt
import androidx.core.hardware.fingerprint.FingerprintManagerCompat
import androidx.fragment.app.FragmentActivity
import com.robotsandpencils.biotest.R
import com.robotsandpencils.biotest.data.pref.PreferenceManager
import com.robotsandpencils.biotest.data.transformer.Async
import com.robotsandpencils.biotest.domain.error.DomainError
import com.robotsandpencils.biotest.domain.error.DomainException
import com.robotsandpencils.biotest.domain.error.ServerException
import com.robotsandpencils.biotest.domain.error.UserFacingException
import com.robotsandpencils.biotest.domain.repository.AuthTokenRepository
import com.robotsandpencils.biotest.domain.usecase.login.RefreshTokenLoginUseCase
import io.reactivex.Completable
import io.reactivex.Single
import io.reactivex.rxkotlin.subscribeBy
import io.reactivex.schedulers.Schedulers
import org.json.JSONObject
import timber.log.Timber
import java.nio.charset.Charset
import java.security.*
import java.security.cert.Certificate
import java.security.spec.MGF1ParameterSpec
import java.security.spec.X509EncodedKeySpec
import java.util.concurrent.Executor
import javax.crypto.Cipher
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.OAEPParameterSpec
import javax.crypto.spec.PSource
import javax.crypto.spec.SecretKeySpec

@RequiresApi(Build.VERSION_CODES.M)
class BiometricsRepository(
    authTokenRepository: AuthTokenRepository,
    private val refreshTokenLoginUseCase: RefreshTokenLoginUseCase,
    private val preferenceManager: PreferenceManager
) {

    /** A couple of useful conversion methods.
     */
    private fun ByteArray.toHex() =
        this.joinToString(separator = "") { it.toInt().and(0xff).toString(16).padStart(2, '0') }

    private fun String.hexStringToByteArray() =
        ByteArray(this.length / 2) { this.substring(it * 2, it * 2 + 2).toInt(16).toByte() }

    class UiThreadExecutor : Executor {
        private val handler = Handler(Looper.getMainLooper())
        override fun execute(command: Runnable?) {
            if (command != null) handler.post(command)
        }
    }

    var biometricPrompt: BiometricPrompt? = null

    init {
        authTokenRepository.refreshTokenUpdated()
            .subscribeOn(Schedulers.computation())
            .observeOn(Schedulers.computation())
            .subscribeBy(
                onNext = { refreshToken ->
                    // Got a new refresh token so need to encrypt it if enrolled.
                    if (areCredentialsStored().blockingGet()) {
                        val username = preferenceManager.getString(PREF_ENROLLED_EMAIL)
                        username?.let {
                            Timber.d("Updating stored biometrics credentials.")
                            storeCredentials(it, refreshToken)
                        }
                    }
                },
                onError = {
                    Timber.e(it)
                })
    }

    /**
     * Perform a biometrics enrollment. This shows the system, or a compat dialog, for fingerprint
     * input, and then encrypts the username and password into shared preferences.
     */
    @RequiresApi(Build.VERSION_CODES.M)
    fun enroll(context: Activity, username: String, refreshToken: String): Completable =
        Completable.create { emitter ->
            val callback = makeAuthenticationCallback(
                onSuccess = {
                    if (emitter.isDisposed) {
                        throw IllegalStateException("Emitter disposed in enroll callback. Indicates unexpected callback reuse.")
                    }
                    biometricPrompt?.cancelAuthentication()
                    try {
                        storeCredentials(username, refreshToken)

                        preferenceManager.putBoolean(PREF_BIOMETRICS_ENROLLED, true)
                        preferenceManager.putString(PREF_ENROLLED_EMAIL, username)

                        // This is what the Settings switch uses
                        preferenceManager.biometricsEnabled.set(true)

                        emitter.onComplete()
                    } catch (ex: Throwable) {
                        emitter.onError(ex)
                    }
                },
                onError = {
                    biometricPrompt?.cancelAuthentication()
                    when (it.errorCode) {
                        BiometricPrompt.ERROR_LOCKOUT -> emitter.onError(DomainException(DomainError.BIOMETRICS_LOCKOUT))
                        BiometricPrompt.ERROR_LOCKOUT_PERMANENT -> emitter.onError(
                            DomainException(
                                DomainError.BIOMETRICS_LOCKOUT_PERMANENT
                            )
                        )
                    }
                })

            val promptInfo = BiometricPrompt.PromptInfo.Builder()
                .setTitle(context.getString(R.string.biometrics_enroll_title))
                .setDescription(context.getString(R.string.biometrics_enroll_subtitle))
                .setNegativeButtonText(context.getString(R.string.biometrics_cancel))
                .build()

            try {
                biometricPrompt =
                        BiometricPrompt(context as FragmentActivity, UiThreadExecutor(), callback)
                            .also {
                                it.authenticate(promptInfo)
                            }
            } catch (ex: Throwable) {
                preferenceManager.putBoolean(PREF_BIOMETRICS_ENROLLED, false)
                emitter.onError(ex)
            }
        }

    private fun storeCredentials(username: String, refreshToken: String) {
        // Generate a symmetric key and encrypt it using the asymmetric cipher
        val symmetricKeyBytes = createSymmetricKey()
        encryptAndSaveSymmetricKey(symmetricKeyBytes, initAsymmetricCipher(Cipher.ENCRYPT_MODE))

        // Make the symmetric cipher to use to save the username and refresh token.
        val symmetricCipher = initSymmetricCipher(Cipher.ENCRYPT_MODE, symmetricKeyBytes)

        encryptAndSaveToken(username, refreshToken, symmetricCipher)
    }

    /**
     * Performs a Biometrics login using either the system dialog or a compat dialog. Takes
     * the encrypted credentials, and decrypts them after a successful biometric login and
     * then uses the refresh token (password) to perform the login.
     */
    @RequiresApi(Build.VERSION_CODES.M)
    fun login(context: Activity): Completable =
        Completable.create { emitter ->

            val promptInfo = BiometricPrompt.PromptInfo.Builder()
                .setTitle(context.getString(R.string.biometrics_login_title))
                .setSubtitle(preferenceManager.getString(PREF_ENROLLED_EMAIL))
                .setNegativeButtonText(context.getString(R.string.biometrics_cancel))
                .build()

            val callback = makeAuthenticationCallback(
                onSuccess = { asymmetricCipher ->

                    if (emitter.isDisposed) {
                        throw IllegalStateException("Emitter disposed in login callback. Indicates unexpected callback reuse.")
                    }

                    biometricPrompt?.cancelAuthentication()
                    try {
                        if (asymmetricCipher != null) {

                            val encryptedSymmetricKey =
                                preferenceManager.getString(PREF_SYMMETRIC_KEY)

                            val symmetricCipher = encryptedSymmetricKey?.let { e ->
                                val symmetricKey =
                                    asymmetricCipher.doFinal(e.hexStringToByteArray())

                                initSymmetricCipher(Cipher.DECRYPT_MODE, symmetricKey)
                            }

                            symmetricCipher?.let { cipher ->
                                val encrypted = preferenceManager.getString(PREF_ENCRYPTED_LOGIN)

                                encrypted?.let { e ->
                                    val clearText = String(
                                        cipher.doFinal(e.hexStringToByteArray()),
                                        DEFAULT_CHARSET
                                    )
                                    val jsonObject = JSONObject(clearText)
                                    val password = jsonObject.getString(PROP_PASSWORD)

                                    refreshTokenLoginUseCase.execute(password, Async).subscribeBy(
                                        onComplete = {
                                            // Not saving the re-encrypted password here because it'll
                                            // be done through the refreshTokenUpdated observable instead.
                                            emitter.onComplete()
                                        },
                                        onError = {
                                            emitter.onError(it)
                                        }
                                    )
                                }
                            }
                        } else {
                            emitter.onError(DomainException(DomainError.BIOMETRICS_AUTHENTICATION_FAILURE))
                        }
                    } catch (ex: Throwable) {
                        emitter.onError(ex)
                    }
                },
                onError = {
                    biometricPrompt?.cancelAuthentication()
                    when (it.errorCode) {
                        BiometricPrompt.ERROR_LOCKOUT -> emitter.onError(DomainException(DomainError.BIOMETRICS_LOCKOUT))
                        BiometricPrompt.ERROR_LOCKOUT_PERMANENT -> emitter.onError(
                            DomainException(
                                DomainError.BIOMETRICS_LOCKOUT_PERMANENT
                            )
                        )
                    }
                })

            try {
                biometricPrompt =
                        BiometricPrompt(context as FragmentActivity, UiThreadExecutor(), callback)
                            .also {
                                it.authenticate(
                                    promptInfo,
                                    makeCryptoObject(initAsymmetricCipher(Cipher.DECRYPT_MODE))
                                )
                            }
            } catch (ex: Throwable) {
                emitter.onError(ex)
            }
        }.onErrorResumeNext {
            when (it) {
                is KeyPermanentlyInvalidatedException, is java.security.UnrecoverableKeyException -> {
                    // Need to delete the key and make a new one later
                    clear().blockingAwait()

                    Completable.error(
                        UserFacingException(
                            context.getString(R.string.biometrics_login_failed_title),
                            context.getString(R.string.biometrics_login_failed_description),
                            it
                        )
                    )
                }
                is ServerException -> {
                    when {
                        // Invalid grant error from the server.
                        it.status == 400 -> Completable.error(
                            UserFacingException(
                                context.getString(R.string.biometrics_login_failed_title),
                                context.getString(R.string.biometrics_login_failed_description),
                                it
                            )
                        )
                        else -> Completable.error(it)
                    }
                }
                else -> Completable.error(it)
            }
        }

    /**
     * Returns true if credentials are stored in the shared preferences.
     */
    fun areCredentialsStored(): Single<Boolean> {
        return Single.just(
            preferenceManager.getBoolean(PREF_BIOMETRICS_ENROLLED, false)
                    && preferenceManager.contains(PREF_SYMMETRIC_KEY)
                    && preferenceManager.contains(PREF_ENCRYPTED_LOGIN)
                    && preferenceManager.contains(PREF_ENROLLED_EMAIL)
                    && preferenceManager.biometricsEnabled.get()
        )
    }

    /**
     * Returns true if the device has biometric hardware.
     */
    fun hasBiometricHardware(context: Activity): Single<Boolean> {
        val hardwareDetected = FingerprintManagerCompat.from(context).isHardwareDetected
        return Single.just(hardwareDetected)
    }

    /**
     * Returns true if the user has added biometric patterns to the device. For example, on
     * a device with a fingerprint sensor, the user has enrolled at least one fingerprint. If
     * this returns false, but hasBiometricHardware returns true, then the user hasn't configured
     * their device for biometrics yet, so we likely shouldn't give them the option to use it.
     */
    fun hasEnrolledBiometricPatterns(context: Activity): Single<Boolean> {
        val fingerprintManagerCompat = FingerprintManagerCompat.from(context)
        val hardwareDetected = fingerprintManagerCompat.isHardwareDetected
        return Single.just(
            hardwareDetected && fingerprintManagerCompat.hasEnrolledFingerprints()
        )
    }

    /**
     * Clears the shared preferences of any biometrics related data.
     */
    fun clear(): Completable =
        Completable.create {
            preferenceManager.remove(PREF_BIOMETRICS_ENROLLED)
            preferenceManager.remove(PREF_SYMMETRIC_KEY)
            preferenceManager.remove(PREF_ENCRYPTED_LOGIN)
            preferenceManager.remove(PREF_ENROLLED_EMAIL)
            preferenceManager.biometricsEnabled.set(false)
            keyStore.deleteEntry(KEY_ALIAS)
            it.onComplete()
        }

    /**
     * Clears biometrics if the user who enrolled is not the username passed in.
     */
    fun clearIfUserMismatch(username: String): Completable =
        if (preferenceManager.getBoolean(PREF_BIOMETRICS_ENROLLED, false)) {
            val enrolledUser = preferenceManager.getString(PREF_ENROLLED_EMAIL)
            when (enrolledUser) {
                username -> Completable.complete()
                else -> clear()
            }
        } else {
            Completable.complete()
        }

    /***
     * PRIVATE METHODS
     */

    private fun encryptAndSaveToken(username: String, refreshToken: String, cipher: Cipher) {
        // Use a simple, built-in json serialization for the data
        val jsonObject = JSONObject()
        jsonObject.put(PROP_EMAIL, username)
        jsonObject.put(PROP_PASSWORD, refreshToken)

        val bytes = jsonObject.toString().toByteArray(DEFAULT_CHARSET)
        val encryptedBytes = cipher.doFinal(bytes)

        encryptedBytes?.let {
            val string = encryptedBytes.toHex()
            preferenceManager.putString(PREF_ENCRYPTED_LOGIN, string)
        }
    }

    private fun encryptAndSaveSymmetricKey(byteArray: ByteArray, asymmetricCipher: Cipher) {
        val encryptedBytes = asymmetricCipher.doFinal(byteArray)
        val string = encryptedBytes.toHex()
        preferenceManager.putString(PREF_SYMMETRIC_KEY, string)
    }

    data class AuthenticationError(val errorCode: Int, val errorString: CharSequence)

    private fun makeAuthenticationCallback(
        onSuccess: (Cipher?) -> Unit,
        onError: (error: AuthenticationError) -> Unit = {}
    ): BiometricPrompt.AuthenticationCallback {
        return object : BiometricPrompt.AuthenticationCallback() {
            override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                onSuccess(result.cryptoObject?.cipher)
            }

            override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                Timber.e("onAuthenticationError $errorCode $errString")
                onError(AuthenticationError(errorCode, errString))
            }

            override fun onAuthenticationFailed() {
                Timber.v("onAuthenticationFailed")
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.M)
    private fun makeKeyPair(keyStore: KeyStore, alias: String): Certificate? {

        val spec = KeyGenParameterSpec.Builder(KEY_ALIAS, KeyProperties.PURPOSE_DECRYPT)
            .setDigests(
                KeyProperties.DIGEST_SHA256,
                KeyProperties.DIGEST_SHA384,
                KeyProperties.DIGEST_SHA512
            )
            .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_RSA_OAEP)
            .setUserAuthenticationRequired(true)

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            spec.setInvalidatedByBiometricEnrollment(true)
        }

        keyPairGenerator.initialize(spec.build())
        keyPairGenerator.generateKeyPair()

        return keyStore.getCertificate(alias)
    }

    @RequiresApi(Build.VERSION_CODES.M)
    private fun makeCryptoObject(cipher: Cipher): BiometricPrompt.CryptoObject =
        BiometricPrompt.CryptoObject(cipher)

    private val keyStore: KeyStore by lazy {
        KeyStore.getInstance("AndroidKeyStore")
    }

    private val keyPairGenerator: KeyPairGenerator by lazy {
        KeyPairGenerator.getInstance("RSA", "AndroidKeyStore")
    }

    private val cipher: Cipher by lazy {
        Cipher.getInstance("RSA/ECB/OAEPWithSHA-256AndMGF1Padding")
    }

    private fun initAsymmetricCipher(mode: Int, c: Cipher = cipher): Cipher {
        keyStore.load(null)

        if (keyStore.getCertificate(KEY_ALIAS) == null) {
            makeKeyPair(keyStore, KEY_ALIAS)
        }

        when (mode) {
            Cipher.ENCRYPT_MODE -> {
                val key: PublicKey = keyStore.getCertificate(KEY_ALIAS).publicKey

                val unrestricted: PublicKey = KeyFactory.getInstance(key.algorithm)
                    .generatePublic(X509EncodedKeySpec(key.encoded))

                val spec = OAEPParameterSpec(
                    "SHA-256",
                    "MGF1",
                    MGF1ParameterSpec.SHA1,
                    PSource.PSpecified.DEFAULT
                )

                c.init(mode, unrestricted, spec)
            }
            else -> {
                val key: PrivateKey = keyStore.getKey(KEY_ALIAS, null) as PrivateKey
                c.init(mode, key)
            }
        }

        return c
    }

    /**
     * Returns the key and iv for an AES/GCM key. 32 bytes for key, 12 bytes for IV.
     */
    private fun createSymmetricKey(): ByteArray {
        val secureRandom = SecureRandom()
        val key = ByteArray(32)
        secureRandom.nextBytes(key)

        val iv = ByteArray(12) //NEVER REUSE THIS IV WITH SAME KEY
        secureRandom.nextBytes(iv)

        return key + iv
    }

    /**
     * See https://proandroiddev.com/security-best-practices-symmetric-encryption-with-aes-in-java-7616beaaade9
     */
    private fun initSymmetricCipher(mode: Int, keyAndIv: ByteArray): Cipher {
        val secretKey = SecretKeySpec(keyAndIv.copyOfRange(0, 32), "AES")

        val cipher = Cipher.getInstance("AES/GCM/NoPadding")
        val parameterSpec =
            GCMParameterSpec(128, keyAndIv.copyOfRange(32, 44)) // 128 bit auth tag length
        cipher.init(mode, secretKey, parameterSpec)

        return cipher
    }

    companion object {
        private const val PREF_BIOMETRICS_ENROLLED = "pref:biometricsEnrolled"
        private const val PREF_ENCRYPTED_LOGIN = "pref:encryptedLogin"
        private const val PREF_ENROLLED_EMAIL = "pref:enrolledEmail"
        private const val PREF_SYMMETRIC_KEY = "pref:symmetricKey"

        private const val KEY_ALIAS = "VaroRSAKey"
        private const val PROP_EMAIL = "email"
        private const val PROP_PASSWORD = "password"

        private val DEFAULT_CHARSET = Charset.forName("UTF-8")
    }
}
