package com.commonsware.todo.repo

import android.app.Application
import androidx.arch.core.executor.testing.InstantTaskExecutorRule
import androidx.test.core.app.ApplicationProvider
import androidx.test.ext.junit.runners.AndroidJUnit4
import com.commonsware.todo.ui.roster.RosterMotor
import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.runBlocking
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.koin.android.ext.koin.with
import org.koin.standalone.StandAloneContext.loadKoinModules
import org.koin.standalone.StandAloneContext.stopKoin
import org.koin.standalone.inject
import org.koin.test.KoinTest

@RunWith(AndroidJUnit4::class)
class RosterMotorTest : KoinTest {
  @get:Rule
  var taskRule = InstantTaskExecutorRule()

  private val repo: ToDoRepository by inject()
  private lateinit var underTest: RosterMotor

  @Before
  fun setUp() {
    stopKoin()
    loadKoinModules(testModule) with ApplicationProvider.getApplicationContext<Application>()
    underTest = RosterMotor(repo)
  }

  @Test
  fun inserts() {
    assertThat(underTest.states.blockingValue?.items).isEmpty()

    val firstModel = ToDoModel(
      description = "Buy a copy of _Exploring Android_",
      notes = "See https://wares.commonsware.com",
      isCompleted = true
    )

    runBlocking { repo.insert(firstModel) }

    assertThat(underTest.states.blockingValue?.items).containsExactly(firstModel)

    val secondModel = ToDoModel(description = "Complete all of the tutorials")
    val thirdModel = ToDoModel(
      description = "Write an app for somebody in my community",
      notes = "Talk to some people at non-profit organizations to see what they need!"
    )

    runBlocking {
      repo.insert(secondModel)
      repo.insert(thirdModel)
    }

    assertThat(underTest.states.blockingValue?.items)
      .containsExactly(firstModel, secondModel, thirdModel)
  }

  @Test
  fun updates() {
    inserts()

    val entities = repo.all().blockingValue!!
    val replacements = entities.map { it.copy(isCompleted = !it.isCompleted) }

    runBlocking { replacements.forEach { repo.update(it) } }

    assertThat(underTest.states.blockingValue?.items)
      .containsExactly(replacements[0], replacements[1], replacements[2])
  }

  @Test
  fun deletes() {
    inserts()

    val entities = repo.all().blockingValue!!

    runBlocking { repo.delete(entities[1]) }

    assertThat(underTest.states.blockingValue?.items).containsExactly(entities[0], entities[2])
  }
}
