package com.commonsware.todo.repo

import android.app.Application
import androidx.arch.core.executor.testing.InstantTaskExecutorRule
import androidx.test.core.app.ApplicationProvider
import androidx.test.ext.junit.runners.AndroidJUnit4
import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.runBlocking
import org.junit.After
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.koin.android.ext.koin.with
import org.koin.standalone.StandAloneContext.loadKoinModules
import org.koin.standalone.StandAloneContext.stopKoin
import org.koin.standalone.inject
import org.koin.test.KoinTest

@RunWith(AndroidJUnit4::class)
class ToDoDatabaseTest : KoinTest {
  @get:Rule var taskRule = InstantTaskExecutorRule()

  private val db: ToDoDatabase by inject()

  @Before
  fun setUp() {
    stopKoin()
    loadKoinModules(testModule) with ApplicationProvider.getApplicationContext<Application>()
  }

  @After
  fun tearDown() {
    db.close()
  }

  @Test
  fun inserts() {
    val store = db.todoStore()

    assertThat(store.all().blockingValue).isEmpty()

    val firstEntity = ToDoEntity(
      description = "Buy a copy of _Exploring Android_",
      notes = "See https://wares.commonsware.com",
      isCompleted = true
    )

    runBlocking { store.save(firstEntity) }

    assertThat(store.all().blockingValue).containsExactly(firstEntity)

    val secondEntity = ToDoEntity(description = "Complete all of the tutorials")
    val thirdEntity = ToDoEntity(
      description = "Write an app for somebody in my community",
      notes = "Talk to some people at non-profit organizations to see what they need!"
    )

    runBlocking { store.save(secondEntity, thirdEntity) }

    assertThat(store.all().blockingValue)
      .containsExactly(firstEntity, secondEntity, thirdEntity)

    listOf(firstEntity, secondEntity, thirdEntity).forEach { entity ->
      assertThat(store.forId(entity.id).blockingValue).isEqualTo(entity)
    }
  }

  @Test
  fun updates() {
    inserts()

    val store = db.todoStore()
    val entities = store.all().blockingValue!!
    val replacements = entities.map { it.copy(isCompleted = !it.isCompleted) }

    runBlocking { store.update(*replacements.toTypedArray()) }

    assertThat(store.all().blockingValue)
      .containsExactly(replacements[0], replacements[1], replacements[2])
  }

  @Test
  fun deletes() {
    inserts()

    val store = db.todoStore()
    val entities = store.all().blockingValue!!

    runBlocking { store.delete(entities[1]) }

    assertThat(store.all().blockingValue)
      .containsExactly(entities[0], entities[2])

    runBlocking { store.deleteAll() }

    assertThat(store.all().blockingValue).isEmpty()
  }
}
