package com.commonsware.todo.repo

import android.app.Application
import androidx.arch.core.executor.testing.InstantTaskExecutorRule
import androidx.test.core.app.ApplicationProvider
import androidx.test.ext.junit.runners.AndroidJUnit4
import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.runBlocking
import org.junit.After
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.koin.android.ext.koin.with
import org.koin.standalone.StandAloneContext.loadKoinModules
import org.koin.standalone.StandAloneContext.stopKoin
import org.koin.standalone.inject
import org.koin.test.KoinTest

@RunWith(AndroidJUnit4::class)
class ToDoRepositoryTest : KoinTest {
  @get:Rule
  var taskRule = InstantTaskExecutorRule()

  private val repo: ToDoRepository by inject()

  @Before
  fun setUp() {
    stopKoin()
    loadKoinModules(testModule) with ApplicationProvider.getApplicationContext<Application>()
  }

  @Test
  fun inserts() {
    assertThat(repo.all().blockingValue).isEmpty()

    val firstModel = ToDoModel(
      description = "Buy a copy of _Exploring Android_",
      notes = "See https://wares.commonsware.com",
      isCompleted = true
    )

    runBlocking { repo.insert(firstModel) }

    assertThat(repo.all().blockingValue).containsExactly(firstModel)

    val secondModel = ToDoModel(description = "Complete all of the tutorials")
    val thirdModel = ToDoModel(
      description = "Write an app for somebody in my community",
      notes = "Talk to some people at non-profit organizations to see what they need!"
    )

    runBlocking {
      repo.insert(secondModel)
      repo.insert(thirdModel)
    }

    assertThat(repo.all().blockingValue)
      .containsExactly(firstModel, secondModel, thirdModel)

    listOf(firstModel, secondModel, thirdModel).forEach { model ->
      assertThat(repo.forId(model.id).blockingValue).isEqualTo(model)
    }
  }

  @Test
  fun updates() {
    inserts()

    val entities = repo.all().blockingValue!!
    val replacements = entities.map { it.copy(isCompleted = !it.isCompleted) }

    runBlocking { replacements.forEach { repo.update(it) } }

    assertThat(repo.all().blockingValue)
      .containsExactly(replacements[0], replacements[1], replacements[2])
  }

  @Test
  fun deletes() {
    inserts()

    val entities = repo.all().blockingValue!!

    runBlocking { repo.delete(entities[1]) }

    assertThat(repo.all().blockingValue).containsExactly(entities[0], entities[2])
  }
}
