package com.example.sample

import android.os.Bundle
import androidx.annotation.MainThread
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.Observer
import androidx.paging.LivePagedListBuilder
import androidx.paging.PagedList
import androidx.recyclerview.selection.SelectionPredicates
import androidx.recyclerview.selection.SelectionTracker
import androidx.recyclerview.selection.StableIdKeyProvider
import androidx.recyclerview.selection.StorageStrategy
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.example.sample.adapter.SampleAdapter
import com.example.sample.adapter.SampleDetailsLookup
import cz.benzina.app.ui.payments.paging.IntDataSourceFactory

class MainActivity : AppCompatActivity() {
    private lateinit var recyclerView: RecyclerView
    private val adapter = SampleAdapter()

    private val livePagedList = LivePagedListBuilder<Int, Int>(IntDataSourceFactory(), PagedList.Config.Builder()
        .setPageSize(PAGE_SIZE)
        .setInitialLoadSizeHint(PAGE_SIZE * 2)
        .setEnablePlaceholders(false)
        .build()
    ).build()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)

        prepareRecyclerView(savedInstanceState)

        livePagedList.observe(this, adapter::submitList)
    }

    private fun prepareRecyclerView(savedInstanceState: Bundle?) {
        recyclerView = findViewById(R.id.list)
        recyclerView.layoutManager = LinearLayoutManager(this)
        recyclerView.addItemDecoration(DividerItemDecoration(this, DividerItemDecoration.VERTICAL))
        recyclerView.adapter = adapter

        adapter.tracker = SelectionTracker.Builder<Long>(
            "selection",
            recyclerView,
            StableIdKeyProvider(recyclerView),
            SampleDetailsLookup(recyclerView),
            StorageStrategy.createLongStorage()
        ).withSelectionPredicate(
            SelectionPredicates.createSelectAnything()
        ).build().apply {
            onRestoreInstanceState(savedInstanceState)
        }

    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        adapter.tracker?.onSaveInstanceState(outState)
    }

    companion object {
        private const val PAGE_SIZE = 10
    }
}

@MainThread
fun <T> LiveData<T>.observe(owner: LifecycleOwner, block: (T) -> Unit) = observe(owner, Observer { block(it) })
