package se.hellsoft.processor

import com.google.auto.service.AutoService
import se.hellsoft.annotation.Screen
import java.io.File
import java.io.FileOutputStream
import javax.annotation.processing.*
import javax.lang.model.SourceVersion
import javax.lang.model.element.ElementKind
import javax.lang.model.element.TypeElement
import javax.tools.Diagnostic

@AutoService(Processor::class)
class ActivityClassProcessor : AbstractProcessor() {
    companion object {
        const val KAPT_KOTLIN_GENERATED_OPTION_NAME = "kapt.kotlin.generated"
    }

    private fun generateActivity(
        packageName: String,
        activityName: String,
        funcName: String,
        outputPath: String
    ) {
        val classData = """
        package $packageName
        
        import android.os.Bundle
        import androidx.appcompat.app.AppCompatActivity
        import androidx.ui.core.setContent

        class $activityName : AppCompatActivity() {
            override fun onCreate(savedInstanceState: Bundle?) {
                super.onCreate(savedInstanceState)
                setContent {
                    ${funcName}(this)
                }
            } 
        }
        """.trimIndent()

        val packagePath = packageName.split(".").joinToString("/")

        val fullPath = File("$outputPath/$packagePath")
        fullPath.mkdirs()
        processingEnv.messager.printMessage(Diagnostic.Kind.NOTE, classData)
        FileOutputStream(File(fullPath,"${activityName}.kt")).buffered().use {
            it.write(classData.toByteArray(Charsets.UTF_8))
        }
    }

    override fun getSupportedOptions(): MutableSet<String> {
        return mutableSetOf(KAPT_KOTLIN_GENERATED_OPTION_NAME)
    }

    override fun getSupportedAnnotationTypes(): MutableSet<String> {
        return mutableSetOf(Screen::class.java.name)
    }

    override fun getSupportedSourceVersion(): SourceVersion = SourceVersion.RELEASE_8

    override fun process(
        annotations: MutableSet<out TypeElement>,
        roundEnv: RoundEnvironment
    ): Boolean {
        roundEnv.getElementsAnnotatedWith(Screen::class.java).forEach { element ->
            if (element.kind == ElementKind.METHOD) {
                val annotation = element.getAnnotation(Screen::class.java)
                val packageName = processingEnv.elementUtils.getPackageOf(element).toString()
                val generatedSourcesRoot: String = processingEnv.options[KAPT_KOTLIN_GENERATED_OPTION_NAME].orEmpty()
                generateActivity(packageName, annotation.name, annotation.function, generatedSourcesRoot)
            }
        }
        return true
    }


}