package com.commonsware.compose.reflection

import android.os.Bundle
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.Composable
import androidx.compose.Composer
import androidx.compose.MutableState
import androidx.compose.currentComposer
import androidx.compose.state
import androidx.ui.core.Modifier
import androidx.ui.core.setContent
import androidx.ui.foundation.AdapterList
import androidx.ui.foundation.Icon
import androidx.ui.foundation.Text
import androidx.ui.graphics.Color
import androidx.ui.layout.Column
import androidx.ui.layout.fillMaxWidth
import androidx.ui.layout.padding
import androidx.ui.material.Button
import androidx.ui.material.DrawerState
import androidx.ui.material.IconButton
import androidx.ui.material.MaterialTheme
import androidx.ui.material.ModalDrawerLayout
import androidx.ui.material.TopAppBar
import androidx.ui.res.vectorResource
import androidx.ui.unit.dp
import kotlin.random.Random

private val MESSAGES = listOf("First Test", "Second Test", "Third Test")

class MainActivity : AppCompatActivity() {
  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)

//    val examples = MESSAGES.map { LambdaExample(it) { SimpleText(it) } }
//    val examples = MESSAGES.map { ReflectionExample(it) }
/*
    val examples = listOf(
      ReflectionFlexExample(MESSAGES[0], "TestOne"),
      ReflectionFlexExample(MESSAGES[1], "TestTwo"),
      ReflectionFlexExample(MESSAGES[2], "TestThree")
    )
*/
/*
    val examples = listOf(
      ReflectionFixedExample(MESSAGES[0], "TestFour"),
      ReflectionFixedExample(MESSAGES[1], "TestFive"),
      ReflectionFixedExample(MESSAGES[2], "TestSix")
    )
*/
/*
    val examples = listOf(
      ReflectionRandomExample(MESSAGES[0], "TestFour"),
      ReflectionRandomExample(MESSAGES[1], "TestFive"),
      ReflectionRandomExample(MESSAGES[2], "TestSix")
    )
*/
/*
    val examples = listOf(
      ReflectionFlexPreviewExample(MESSAGES[0], "TestOne"),
      ReflectionFlexPreviewExample(MESSAGES[1], "TestTwo"),
      ReflectionFlexPreviewExample(MESSAGES[2], "TestThree")
    )
*/
    val examples = listOf(
      ReflectionFixedPreviewExample(MESSAGES[0], "TestFour"),
      ReflectionFixedPreviewExample(MESSAGES[1], "TestFive"),
      ReflectionFixedPreviewExample(MESSAGES[2], "TestSix")
    )

    setContent {
      MainContent(examples)
    }
  }
}

@Composable
fun MainContent(items: List<Example>) {
  val (drawerState, onDrawerStateChange) = state { DrawerState.Closed }
  val contentState = state { items.first() }

  MaterialTheme {
    ModalDrawerLayout(
      drawerState = drawerState,
      onStateChange = onDrawerStateChange,
      gesturesEnabled = true,
      drawerContent = {
        AppDrawer(items, contentState) {
          onDrawerStateChange(DrawerState.Closed)
        }
      },
      bodyContent = {
        Content(contentState) { onDrawerStateChange(DrawerState.Opened) }
      }
    )
  }
}

@Composable
private fun AppDrawer(
  items: List<Example>,
  contentState: MutableState<Example>,
  closeDrawer: () -> Unit
) {
  AdapterList(items) {
    Button(
      onClick = { contentState.value = it; closeDrawer() },
      modifier = Modifier.padding(8.dp).fillMaxWidth()
    ) {
      Text(it.title)
    }
  }
}

@Composable
fun Content(
  contentState: MutableState<Example>,
  openDrawer: () -> Unit
) {
  Column {
    TopAppBar(
      title = {
        Text(text = "AppBar")
      },
      contentColor = Color.White,
      navigationIcon = {
        IconButton(onClick = { openDrawer() }) {
          Icon(vectorResource(id = R.drawable.ic_menu))
        }
      }
    )

    contentState.value.render(composer = currentComposer)
  }
}

@Composable
fun SimpleText(message: String) {
  Text(message, modifier = Modifier.padding(8.dp))
}

@Composable
fun TestOne() {
  SimpleText("TestOne")
}

@Composable
fun TestTwo() {
  SimpleText("TestTwo")
}

@Composable
fun TestThree() {
  SimpleText("TestThree")
}

@Composable
fun TestFour(unused: Int) {
  SimpleText("TestFour")
}

@Composable
fun TestFive(unused: Int) {
  SimpleText("TestFive")
}

@Composable
fun TestSix(unused: Int) {
  SimpleText("TestSix")
}

interface Example {
  val title: String

  @Composable
  fun render(composer: Composer<*>)
}

data class LambdaExample(
  override val title: String,
  private val content: @Composable() () -> Unit
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    content()
  }
}

data class ReflectionExample(
  override val title: String
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    val method = Class.forName("com.commonsware.compose.reflection.MainActivityKt")
      .methods
      .find { it.name == "SimpleText" }

    method?.apply {
      isAccessible = true
      invoke(null, title, composer, 0, 0)
    }
  }
}

data class ReflectionFlexExample(
  override val title: String,
  private val function: String
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    val method = Class.forName("com.commonsware.compose.reflection.MainActivityKt")
      .methods
      .find { it.name == function }

    method?.apply {
      isAccessible = true
      invoke(null, title, composer, 0, 0)
    }
  }
}

data class ReflectionFlexPreviewExample(
  override val title: String,
  private val function: String
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    invokeComposableViaReflection(
      "com.commonsware.compose.reflection.MainActivityKt",
      function,
      composer
    )
  }
}

data class ReflectionFixedExample(
  override val title: String,
  private val function: String
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    val method = Class.forName("com.commonsware.compose.reflection.MainActivityKt")
      .methods
      .find { it.name == function }

    method?.apply {
      isAccessible = true
      invoke(null, 1337, composer, 0, 0)
    }
  }
}


data class ReflectionFixedPreviewExample(
  override val title: String,
  private val function: String
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    invokeComposableViaReflection(
      "com.commonsware.compose.reflection.MainActivityKt",
      function,
      composer,
      1337
    )
  }
}

data class ReflectionRandomExample(
  override val title: String,
  private val function: String
) : Example {
  @Composable
  override fun render(composer: Composer<*>) {
    val method = Class.forName("com.commonsware.compose.reflection.MainActivityKt")
      .methods
      .find { it.name == function }

    method?.apply {
      isAccessible = true
      invoke(null, Random.nextInt(), composer, 0, 0)
    }
  }
}