package az.sample.fingerprintchecher

import android.os.Build
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyPermanentlyInvalidatedException
import android.security.keystore.KeyProperties
import androidx.annotation.RequiresApi
import androidx.biometric.BiometricPrompt
import java.io.IOException
import java.lang.Exception
import java.security.*
import java.security.cert.CertificateException
import javax.crypto.Cipher
import javax.crypto.KeyGenerator
import javax.crypto.NoSuchPaddingException
import javax.crypto.SecretKey

object CipherUtils {
    private const val fingerPrintKey: String = "finger_print_key"


    private var mCipher: Cipher? = null
    private var mKeyStore: KeyStore? = null
    private var mKeyGenerator: KeyGenerator? = null

    private var mCryptoObject: BiometricPrompt.CryptoObject? = null

    private var listener: CipherListener? = null

    interface CipherListener {

        fun onCipherInitFail(e : Exception)
        fun onCipherInitSuccessfully(cryptoObject: BiometricPrompt.CryptoObject)
    }

    @RequiresApi(Build.VERSION_CODES.M)
    fun init(listener: CipherListener? = null) {
        this.listener = listener
        initKeyStore()

        if (getKey(fingerPrintKey) == null) {
            generateKey(fingerPrintKey)
        }
        val isSuccessfullyInit = cipherInit(fingerPrintKey, Cipher.ENCRYPT_MODE)

        if (isSuccessfullyInit) {
            mCryptoObject = BiometricPrompt.CryptoObject(mCipher!!)
            listener?.onCipherInitSuccessfully(mCryptoObject!!)
        }
    }


    fun initKeyStore() {
        mKeyStore = KeyStore.getInstance("AndroidKeyStore")
        mKeyStore?.load(null)
    }


    private fun getKey(keyName: String): Key? {
        return mKeyStore?.getKey(keyName, null)
    }

    @RequiresApi(Build.VERSION_CODES.M)
    private fun cipherInit(keyName: String, mode: Int): Boolean {
        try {
            mCipher = Cipher.getInstance(
                KeyProperties.KEY_ALGORITHM_AES + "/"
                        + KeyProperties.BLOCK_MODE_CBC + "/"
                        + KeyProperties.ENCRYPTION_PADDING_PKCS7
            )
        } catch (e: NoSuchAlgorithmException) {
            listener?.onCipherInitFail(e)

        } catch (e: NoSuchPaddingException) {
            listener?.onCipherInitFail(e)
        }

        try {
            val key = mKeyStore!!.getKey(keyName, null) as SecretKey
            mCipher!!.init(Cipher.ENCRYPT_MODE, key)
            return true
        } catch (e: KeyPermanentlyInvalidatedException) {
            listener?.onCipherInitFail(e)
        } catch (e: KeyStoreException) {
            listener?.onCipherInitFail(e)
        } catch (e: CertificateException) {
            listener?.onCipherInitFail(e)
        } catch (e: UnrecoverableKeyException) {
            listener?.onCipherInitFail(e)
        } catch (e: IOException) {
            listener?.onCipherInitFail(e)
        } catch (e: NoSuchAlgorithmException) {
            listener?.onCipherInitFail(e)
        } catch (e: InvalidKeyException) {
            listener?.onCipherInitFail(e)
        }
        return false
    }


    @RequiresApi(Build.VERSION_CODES.M)
    fun generateKey(key: String = fingerPrintKey) {
        try {
            mKeyGenerator = KeyGenerator.getInstance(
                "AES",
                "AndroidKeyStore"
            )
        } catch (e: NoSuchAlgorithmException) {
            throw RuntimeException(
                "Failed to get KeyGenerator instance", e
            )
        } catch (e: NoSuchProviderException) {
            throw RuntimeException("Failed to get KeyGenerator instance", e)
        }

        try {
            mKeyGenerator!!.init(
                KeyGenParameterSpec.Builder(
                    key,
                    KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
                )
                    .setBlockModes(KeyProperties.BLOCK_MODE_CBC)
                    .setUserAuthenticationRequired(true)
                    .setEncryptionPaddings(
                        KeyProperties.ENCRYPTION_PADDING_PKCS7
                    )
                    .build()
            )
            mKeyGenerator!!.generateKey()
        } catch (e: NoSuchAlgorithmException) {
            throw RuntimeException(e)
        } catch (e: InvalidAlgorithmParameterException) {
            throw RuntimeException(e)
        } catch (e: CertificateException) {
            throw RuntimeException(e)
        } catch (e: IOException) {
            throw RuntimeException(e)
        }
    }

}