/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.samples.donuttracker.coffee

import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.app.NotificationManagerCompat
import androidx.fragment.app.Fragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.ViewModelProvider
import androidx.navigation.findNavController
import androidx.navigation.fragment.findNavController
import com.android.samples.donuttracker.coffee.databinding.CoffeeListBinding
import com.android.samples.donuttracker.core.storage.SnackDatabase

/**
 * Fragment containing the RecyclerView which shows the current list of coffees being tracked.
 */
class CoffeeList : Fragment() {

    private lateinit var coffeeListViewModel: CoffeeListViewModel

    private val adapter = CoffeeListAdapter(
        onEdit = { coffee ->
            findNavController().navigate(
                CoffeeListDirections.actionCoffeeListToCoffeeEntryDialogFragment(coffee.id)
            )
        },
        onDelete = { coffee ->
            NotificationManagerCompat.from(requireContext()).cancel(coffee.id.toInt())
            coffeeListViewModel.delete(coffee)
        }
    )

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {

        val navController = findNavController()

        // After a configuration change or process death, the currentBackStackEntry
        // points to the dialog destination, so you must use getBackStackEntry()
        // with the specific ID of your destination to ensure we always
        // get the right NavBackStackEntry
        val navBackStackEntry = navController.getBackStackEntry(R.id.coffeeList)

        // Create our observer and add it to the NavBackStackEntry's lifecycle
        val observer = LifecycleEventObserver { _, event ->
            if (event == Lifecycle.Event.ON_RESUME
                && navBackStackEntry.savedStateHandle.contains("key")) {
                val result = navBackStackEntry.savedStateHandle.get<String>("key")
                Log.d("backStackEntry result", "result = $result")
                // Do something with the result
                navController.popBackStack()
            }
        }
        navBackStackEntry.lifecycle.addObserver(observer)

        // As addObserver() does not automatically remove the observer, we
        // call removeObserver() manually when the view lifecycle is destroyed
        viewLifecycleOwner.lifecycle.addObserver(LifecycleEventObserver { _, event ->

            when (event) {
                Lifecycle.Event.ON_CREATE -> Log.d("LifecycleEventObserver", "ON_CREATE")
                Lifecycle.Event.ON_START -> Log.d("LifecycleEventObserver", "ON_START")
                Lifecycle.Event.ON_RESUME -> Log.d("LifecycleEventObserver", "ON_RESUME")
                Lifecycle.Event.ON_PAUSE -> Log.d("LifecycleEventObserver", "ON_PAUSE")
                Lifecycle.Event.ON_STOP -> Log.d("LifecycleEventObserver", "ON_STOP")
                Lifecycle.Event.ON_DESTROY -> Log.d("LifecycleEventObserver", "ON_DESTROY")
                Lifecycle.Event.ON_ANY -> Log.d("LifecycleEventObserver", "ON_ANY")
            }

            if (event == Lifecycle.Event.ON_DESTROY) {
                navBackStackEntry.lifecycle.removeObserver(observer)
            }
        })


        return CoffeeListBinding.inflate(inflater, container, false).root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        val binding = CoffeeListBinding.bind(view)
        val coffeeDao = SnackDatabase.getDatabase(requireContext()).coffeeDao()

        coffeeListViewModel = ViewModelProvider(this, CoffeeViewModelFactory(coffeeDao))
            .get(CoffeeListViewModel::class.java)

        coffeeListViewModel.coffeeList.observe(viewLifecycleOwner) { coffees ->
            adapter.submitList(coffees)
        }

        binding.recyclerView.adapter = adapter

        binding.fab.setOnClickListener { fabView ->
            fabView.findNavController().navigate(
                CoffeeListDirections.actionCoffeeListToCoffeeEntryDialogFragment()
            )
        }

        binding.fab2.setOnClickListener { fabView ->
            fabView.findNavController().navigate(
                CoffeeListDirections.actionCoffeeListToCoffeeList2Fragment()
            )
        }
    }
}
