package com.example.roomksprepro

import androidx.room.Dao
import androidx.room.Database
import androidx.room.Entity
import androidx.room.Insert
import androidx.room.PrimaryKey
import androidx.room.Query
import androidx.room.RoomDatabase
import androidx.room.TypeConverter
import androidx.room.TypeConverters

@Database(
    version = 1,
    entities = [
        User::class,
    ],
)
@TypeConverters(DbConversions::class)
abstract class AppDatabase : RoomDatabase() {
    abstract val user: UserDao
}

@Dao
interface UserDao {

    // Incorrect: doesn't use DbConversions.toString(Country), instead generates a converter __Country_enumToString that uses Enum.name (FRANCE instead of FR)
    @Insert
    fun insert(user: User): Long

    // Correct: uses DbConversions.toString(Country)
    @Query("UPDATE user SET country1 = :country1 WHERE id = :id")
    fun setCountry1(id: Long, country1: Country)

    // Incorrect: doesn't use DbConversions.toString(Country), instead generates a converter __Country_enumToString that uses Enum.name (FRANCE instead of FR)
    @Query("UPDATE user SET country2 = :country2 WHERE id = :id")
    fun setCountry2(id: Long, country2: Country?)

    // Incorrect: uses DbConversions.toString(Country) for both fields, but passes String? in for country2, so can cause an NPE at runtime
    @Query("SELECT * FROM user WHERE id = :id")
    fun getById(id: Long): User?

    // Correct: uses the nullable type converter
    @Query("UPDATE user SET country2 = :country2 WHERE id = :id")
    @TypeConverters(DbConversionsNull::class)
    fun setCountry2WithTC(id: Long, country2: Country?)

    // Correct: uses DbConversions.toString(Country)
    @Query("SELECT country1 FROM user WHERE id = :id")
    fun getCountry1(id: Long): Country

    // Incorrect: uses DbConversions.toString(Country) for both fields, but passes String? in for country2, so can cause an NPE at runtime
    @Query("SELECT country2 FROM user WHERE id = :id")
    fun getCountry2(id: Long): Country?
}

@Entity(tableName = "user")
data class User(
    @PrimaryKey
    val id: Long,
    val country1: Country,
    val country2: Country?,
)

enum class Country(val countryCode: String) {
    UNITED_KINGDOM("GB"),
    FRANCE("FR"),
}

class DbConversions {

    @TypeConverter
    fun toString(country: Country): String {
        return country.countryCode
    }

    // This won't compile within the same type converter class; error is:
    // >> Multiple methods define the same conversion. Conflicts with these: CustomTypeConverter(enclosingClass=DbConversions, isEnclosingClassKotlinObject=false, method=toStringNull, from=Country?, to=String?, isProvidedConverter=false)
    // (Room considers `Country` and `Country?` to be different types as params/properties, but the same type as type converters?)
//    @TypeConverter
//    fun toStringNull(country: Country?): String? {
//        return country?.countryCode
//    }

    @TypeConverter
    fun toCountry(s: String): Country {
        return Country.values().find { it.countryCode == s }
            ?: throw IllegalArgumentException("Country code '$s' not found")
    }
}

// Can work around the bug by making a second type converter with nullable versions of the methods,
// and applying it where needed with @TypeConverters(DbConversionsNull::class).
// This is error-prone - if you forget to apply it, Room will quietly automatically generate enum type converters that use Enum.name.
class DbConversionsNull {
    @TypeConverter
    fun toStringNull(country: Country?): String? {
        return country?.countryCode
    }
}
