package tm.alashow.navigationcomposebug

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.animation.Crossfade
import androidx.compose.foundation.layout.*
import androidx.compose.material.*
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Search
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material.icons.outlined.Search
import androidx.compose.material.icons.outlined.Settings
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.graphics.vector.rememberVectorPainter
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.navigation.NavController
import androidx.navigation.NavDestination.Companion.hierarchy
import androidx.navigation.NavGraph.Companion.findStartDestination
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import androidx.navigation.navigation
import tm.alashow.navigationcomposebug.ui.theme.NavigationComposeBugTheme
import java.lang.Math.abs
import kotlin.random.Random

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContent {
            NavigationComposeBugTheme {
                Home()
            }
        }
    }
}

@Composable
fun Screen(name: String) {
    Column(modifier = Modifier.fillMaxSize(), horizontalAlignment = Alignment.CenterHorizontally, verticalArrangement = Arrangement.Center) {
        Text(text = name, style = MaterialTheme.typography.h4)
        Text(abs(Random.nextInt()).toString())
    }
}


const val SearchTab = "SearchRoot"
const val SearchScreen = "SearchScreen"
const val SettingsTab = "SettingsRoot"
const val SettingsScreen = "SettingsScreen"

@Composable
internal fun Home(
    navController: NavHostController = rememberNavController(),
    scaffoldState: ScaffoldState = rememberScaffoldState(),
) {

    Scaffold(
        scaffoldState = scaffoldState,
        bottomBar = {
            val currentSelectedItem by navController.currentScreenAsState()

            HomeBottomNavigation(
                selectedNavigation = currentSelectedItem,
                onNavigationSelected = { selected ->
                    navController.navigate(selected) {
                        launchSingleTop = true
                        restoreState = true

                        popUpTo(navController.graph.findStartDestination().id) {
                            saveState = true
                        }
                    }
                },
                modifier = Modifier.fillMaxWidth()
            )
        }
    ) {
        Box(Modifier.fillMaxSize()) {
            AppNavigation(navController)
        }
    }
}

@Composable
fun AppNavigation(navController: NavHostController) {
    NavHost(
        navController = navController,
        startDestination = SearchTab
    ) {
        navigation(
            route = SearchTab,
            startDestination = SearchScreen
        ) {
            composable(SearchScreen) {
                Screen(name = SearchScreen)
            }
        }
        navigation(
            route = SettingsTab,
            startDestination = SettingsScreen
        ) {
            composable(SettingsScreen) {
                Screen(name = SettingsScreen)
            }
        }
    }
}

@Stable
@Composable
private fun NavController.currentScreenAsState(): State<String> {
    val selectedItem = remember { mutableStateOf<String>(SearchTab) }

    DisposableEffect(this) {
        val listener = NavController.OnDestinationChangedListener { _, destination, _ ->
            when {
                destination.hierarchy.any { it.route == SearchTab } -> {
                    selectedItem.value = SearchTab
                }
                destination.hierarchy.any { it.route == SettingsTab } -> {
                    selectedItem.value = SettingsTab
                }
            }
        }
        addOnDestinationChangedListener(listener)

        onDispose {
            removeOnDestinationChangedListener(listener)
        }
    }

    return selectedItem
}

@Composable
internal fun HomeBottomNavigation(
    selectedNavigation: String,
    onNavigationSelected: (String) -> Unit,
    modifier: Modifier = Modifier
) {
    Surface(
        elevation = 8.dp,
        modifier = modifier
    ) {
        Row(
            Modifier
                .fillMaxWidth()
                .height(56.dp),
            horizontalArrangement = Arrangement.SpaceBetween,
        ) {
            HomeBottomNavigationItem(
                label = SearchTab,
                contentDescription = SearchTab,
                selected = selectedNavigation == SearchTab,
                onClick = { onNavigationSelected(SearchTab) },
                painter = rememberVectorPainter(Icons.Outlined.Search),
                selectedPainter = rememberVectorPainter(Icons.Filled.Search),
            )
            HomeBottomNavigationItem(
                label = SettingsTab,
                contentDescription = SettingsTab,
                selected = selectedNavigation == SettingsTab,
                onClick = { onNavigationSelected(SettingsTab) },
                painter = rememberVectorPainter(Icons.Outlined.Settings),
                selectedPainter = rememberVectorPainter(Icons.Filled.Settings),
            )
        }
    }
}

@Composable
private fun RowScope.HomeBottomNavigationItem(
    selected: Boolean,
    selectedPainter: Painter? = null,
    painter: Painter,
    contentDescription: String,
    label: String,
    onClick: () -> Unit,
) {
    BottomNavigationItem(
        icon = {
            if (selectedPainter != null) {
                Crossfade(targetState = selected) { selected ->
                    Icon(
                        painter = if (selected) selectedPainter else painter,
                        contentDescription = contentDescription
                    )
                }
            } else {
                Icon(
                    painter = painter,
                    contentDescription = contentDescription
                )
            }
        },
        label = { Text(label) },
        selected = selected,
        onClick = onClick,
        selectedContentColor = MaterialTheme.colors.secondary,
        unselectedContentColor = MaterialTheme.colors.onSurface,
    )
}
