package com.example.desugaringtest.time

import android.content.SharedPreferences
import android.os.SystemClock
import java.time.DayOfWeek
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.OffsetDateTime
import java.time.ZoneId
import java.time.ZoneOffset
import java.time.ZonedDateTime
import kotlin.time.Duration.Companion.seconds

const val TIME_NOT_SET = -1L

@Volatile private var globalServerTimeMillisOverride: Long? = null

private val currentServerTimeMillis
  get() = globalServerTimeMillisOverride ?: 1L.let { time ->
    if(time == TIME_NOT_SET) {
      TIME_NOT_SET
    }
    else {
      time + SystemClock.elapsedRealtime()
    }
  }

interface CurrentTimeHelper {
  var serverTimeMillisOverride: Long?

  fun initialize()

  suspend fun awaitCurrentServerTime(): ZonedDateTime
  suspend fun awaitCurrentServerTimeMillis(): Long

  companion object {
    private fun serverZonedDateTime(inUtc: Boolean) = currentServerTimeMillis.let { time ->
      if(time == TIME_NOT_SET) {
        null
      }
      else {
        when {
          inUtc -> time.toUtcDateTime()
          else -> time.toZonedDateTime()
        }
      }
    }

    fun serverNowOrNull() = serverZonedDateTime(inUtc = false)
    fun serverNowUtcOrNull() = serverZonedDateTime(inUtc = true)

    fun serverNowOrDeviceNow() = serverNowOrNull() ?: ZonedDateTime.now()
    fun serverNowUtcOrDeviceNowUtc() = serverNowUtcOrNull() ?: ZonedDateTime.now(ZoneOffset.UTC)
  }
}

fun nowUtc(): ZonedDateTime = ZonedDateTime.now(ZoneOffset.UTC)
fun nowZoned(): ZonedDateTime = ZonedDateTime.now(ZoneId.systemDefault())

fun ZonedDateTime.toMillis() = toInstant().toEpochMilli()
fun LocalDateTime.toMillis() = atZone(ZoneId.systemDefault()).toMillis()
fun LocalDate.toMillis() = atStartOfDay(ZoneId.systemDefault()).toMillis()

fun Long.toUtcLocalDate(): LocalDate = Instant.ofEpochMilli(this).atZone(ZoneOffset.UTC).toLocalDate()

fun Long.toUtcDateTime(): ZonedDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(this), ZoneOffset.UTC)
fun Double.toUtcDateTime(): ZonedDateTime = (this * 1000.0).toLong().toUtcDateTime()

fun Long.toZonedDateTime(): ZonedDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(this), ZoneId.systemDefault())
fun Double.toZonedDateTime(): ZonedDateTime = (this * 1000.0).toLong().toZonedDateTime()

fun Long.toLocalDateTime(): LocalDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(this), ZoneId.systemDefault())

fun Long.toOffsetDateTime(offsetSeconds: Int): OffsetDateTime =
  Instant
    .ofEpochMilli(this)
    .atOffset(ZoneOffset.ofTotalSeconds(offsetSeconds))

fun Long.toOffsetLocalDateTime(offsetSeconds: Int): LocalDateTime =
  toOffsetDateTime(offsetSeconds).toLocalDateTime()

fun Long.toZonedDateTime(zone: String): ZonedDateTime =
  ZonedDateTime.ofInstant(Instant.ofEpochMilli(this), ZoneId.of(zone))

fun Double.toZonedDateTime(zone: String): ZonedDateTime =
  (this * 1000.0).toLong().toZonedDateTime(zone)

fun Long.toZonedDateTime(zone: ZoneId): ZonedDateTime =
  ZonedDateTime.ofInstant(Instant.ofEpochMilli(this), zone)

fun Double.toZonedDateTime(zone: ZoneId): ZonedDateTime =
  (this * 1000.0).toLong().toZonedDateTime(zone)

val LocalDate.daysToWeekend
  get() =
    when(dayOfWeek) {
      DayOfWeek.MONDAY -> 5L
      DayOfWeek.TUESDAY -> 4L
      DayOfWeek.WEDNESDAY -> 3L
      DayOfWeek.THURSDAY -> 2L
      DayOfWeek.FRIDAY -> 1L
      DayOfWeek.SATURDAY -> 0L
      DayOfWeek.SUNDAY -> 0L
      null -> 0L // NOTE because JAVA
    }

val LocalDate.daysToEndOfWeekend
  get() =
    when(dayOfWeek) {
      DayOfWeek.MONDAY -> 7L
      DayOfWeek.TUESDAY -> 6L
      DayOfWeek.WEDNESDAY -> 5L
      DayOfWeek.THURSDAY -> 4L
      DayOfWeek.FRIDAY -> 3L
      DayOfWeek.SATURDAY -> 2L
      DayOfWeek.SUNDAY -> 1L
      null -> 0L // NOTE because JAVA
    }
