package com.andreirozov.sample.ui

import android.util.Log
import androidx.compose.animation.AnimatedContentScope
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.animation.core.tween
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Menu
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.navigation.NavHostController
import com.andreirozov.sample.R
import com.andreirozov.sample.ui.screen1.SuccessState
import com.andreirozov.sample.ui.standings.screen2.AnotherScreen
import com.andreirozov.sample.utils.DrawerNavigationItem
import com.google.accompanist.navigation.animation.AnimatedNavHost
import com.google.accompanist.navigation.animation.composable

@OptIn(ExperimentalAnimationApi::class, ExperimentalMaterial3Api::class)
@Composable
fun AppNavGraph(navController: NavHostController, openDrawer: () -> Unit) {
    AnimatedNavHost(
        navController = navController,
        startDestination = DrawerNavigationItem.Screen1.route,
        enterTransition = {
            slideIntoContainer(
                AnimatedContentScope.SlideDirection.Left,
                animationSpec = tween(200)
            )
        },
        exitTransition = {
            slideOutOfContainer(
                AnimatedContentScope.SlideDirection.Left,
                animationSpec = tween(200)
            )
        },
        popEnterTransition = {
            slideIntoContainer(
                AnimatedContentScope.SlideDirection.Right,
                animationSpec = tween(200)
            )
        },
        popExitTransition = {
            slideOutOfContainer(
                AnimatedContentScope.SlideDirection.Right,
                animationSpec = tween(200)
            )
        }
    ) {
        // Navigate to standings screen
        composable(
            route = DrawerNavigationItem.Screen2.route,
            content = {
                Log.e("Something", "Standing back stack entry state is ${it.lifecycle.currentState}")
                val scrollBehavior = TopAppBarDefaults.enterAlwaysScrollBehavior(
                    rememberTopAppBarState()
                )

                Scaffold(
                    topBar = {
                        TopAppBar(
                            title = { Text(text = stringResource(id = R.string.screen2)) },
                            scrollBehavior = scrollBehavior,
                            navigationIcon = {
                                IconButton(
                                    onClick = { openDrawer() },
                                    content = {
                                        Icon(
                                            imageVector = Icons.Default.Menu,
                                            contentDescription = stringResource(R.string.open_drawer)
                                        )
                                    }
                                )
                            }
                        )
                    },
                    content = { innerPadding ->
                        Log.e("Something", "Standing back stack entry state in Scaffold ${it.lifecycle.currentState}")
                        Column(
                            modifier = Modifier.padding(innerPadding).fillMaxSize(),
                            content = {
                                AnotherScreen(scrollBehavior = scrollBehavior)
                            }
                        )
                    }
                )
            }
        )

        // Navigate to scores screen
        composable(
            route = DrawerNavigationItem.Screen1.route,
            content = {
                Scaffold(
                    topBar = {
                        TopAppBar(
                            title = {
                                Text(text = stringResource(id = R.string.screen1))
                            },
                            navigationIcon = {
                                IconButton(
                                    onClick = { openDrawer() },
                                    content = {
                                        Icon(
                                            imageVector = Icons.Default.Menu,
                                            contentDescription = stringResource(R.string.open_drawer)
                                        )
                                    }
                                )
                            }
                        )
                    },
                    content = { innerPadding ->
                        SuccessState(innerPadding = innerPadding)
                    }
                )
            }
        )
    }
}