package com.andreirozov.sample.ui

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.BackHandler
import androidx.activity.compose.setContent
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.material3.DrawerValue
import androidx.compose.material3.ModalNavigationDrawer
import androidx.compose.material3.rememberDrawerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.navigation.compose.currentBackStackEntryAsState
import com.andreirozov.sample.ui.theme.MyTheme
import com.andreirozov.sample.utils.DrawerNavigationItem
import com.google.accompanist.navigation.animation.rememberAnimatedNavController
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.launch

@AndroidEntryPoint
class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        setContent {
            MyApp()
        }
    }
}

@OptIn(ExperimentalAnimationApi::class)
@Composable
private fun MyApp() {
    MyTheme {
        // Create navController
        val navController = rememberAnimatedNavController()

        // Subscribe to navBackStackEntry, required to get current route
        val navBackStackEntry by navController.currentBackStackEntryAsState()

        // Current route for drawer
        val currentRoute =
            navBackStackEntry?.destination?.route ?: DrawerNavigationItem.Screen1.route

        // Drawer state
        val drawerState = rememberDrawerState(initialValue = DrawerValue.Closed)

        // Coroutine scope for open/close drawer
        val coroutineScope = rememberCoroutineScope()

        // If drawer is open or animation running catch back press and close drawer
        BackHandler(drawerState.isOpen || drawerState.isAnimationRunning) {
            coroutineScope.launch {
                drawerState.close()
            }
        }

        ModalNavigationDrawer(
            drawerState = drawerState,
            gesturesEnabled = drawerGestureEnabled(currentRoute = currentRoute),
            drawerContent = {
                AppDrawer(
                    currentRoute = currentRoute,
                    navController = navController,
                    closeDrawer = {
                        coroutineScope.launch {
                            drawerState.close()
                        }
                    }
                )
            },
            content = {
                AppNavGraph(
                    navController = navController,
                    openDrawer = {
                        coroutineScope.launch {
                            drawerState.open()
                        }
                    }
                )
            }
        )
    }
}

// If current screen is one of the main screens, drawer could be open by swipe, otherwise drawer disabled
private fun drawerGestureEnabled(currentRoute: String): Boolean {
    return currentRoute == "standings"
            || currentRoute == "scores"
}