package com.pacmac.navigationlibsample

import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material3.Button
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.navigation.NavHost
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import com.pacmac.navigationlibsample.ui.theme.NavigationLibSampleTheme

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            NavigationLibSampleTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { padding ->

                    val navHostController = rememberNavController()

                    NavHost(
                        navController = navHostController,
                        startDestination = "hello",
                        modifier = Modifier.padding(padding)
                    ) {

                        composable("hello") {
                            Hello("pacmac", onClick = {
                                navHostController.navigate("world")
                            })
                        }
                        composable("world") {
                            World()
                        }
                    }
                }
            }
        }
    }
}

@Composable
fun Hello(name: String, modifier: Modifier = Modifier, onClick:()->Unit) {
    attachLifeCycleObserver() {
        println("PACMAC --- on DESTROY -- HELLO")
    }
    Column(Modifier.padding(16.dp)) {
        Text(
        text = "Hello $name!",
        modifier = modifier
        )
        Spacer(modifier = Modifier.size(32.dp))
        Button(onClick = onClick) {
            Text(text = "Go to next screen")
        }
    }
}

@Composable
fun World(modifier: Modifier = Modifier) {
    attachLifeCycleObserver() {
        println("PACMAC --- on DESTROY -- WORLD")
    }
    Text(
        text = "Hello World!",
        modifier = modifier
    )
}

@Composable
fun attachLifeCycleObserver(onDestroy:() -> Unit) {

    val lifecycle = LocalLifecycleOwner.current.lifecycle
    DisposableEffect(lifecycle) {
        val eventObserver = LifecycleEventObserver { source, event ->
            println("PACMAC -- event:${event.name}  targetState:${event.targetState.name} source:${source.toString()}")
            if (event == Lifecycle.Event.ON_DESTROY) {
                onDestroy()
            }
        }
        println("PACMAC -- adding observer:: $eventObserver")
        lifecycle.addObserver(eventObserver)
        onDispose {
            println("PACMAC -- removing observer: $eventObserver")
            lifecycle.removeObserver(eventObserver)
        }
    }
}