package com.vulpeszerda.navigation.component.sample

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.Service
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.IBinder
import androidx.core.app.NotificationCompat
import androidx.core.app.ServiceCompat
import androidx.core.content.ContextCompat
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import timber.log.Timber
import kotlin.time.Duration.Companion.seconds

class ForegroundService : Service() {

    private val coroutineScope = CoroutineScope(SupervisorJob() + Dispatchers.Main.immediate)

    private val notificationManager: NotificationManager by lazy {
        getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
    }

    private val receiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context?, intent: Intent?) {
            stopSelf()
        }
    }

    override fun onCreate() {
        super.onCreate()
        LocalBroadcastManager.getInstance(this).registerReceiver(receiver, IntentFilter(ACTION_STOP))

        startForeground()
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        super.onStartCommand(intent, flags, startId)

        // start foreground again for sure
        startForeground()
        coroutineScope.launch {
            while (true) {
                Timber.d("[ForegroundService] service running..")
                delay(3.seconds)
            }
        }
        return START_NOT_STICKY
    }

    override fun onBind(intent: Intent?): IBinder? {
        throw UnsupportedOperationException("Service should be started with `onStartCommand`")
    }

    private fun startForeground() {
        setupChannel()
        startForeground(
            NOTIFICATION_ID,
            NotificationCompat.Builder(this, CHANNEL_ID)
                .apply {
                    setContentTitle("NavSample")
                    setContentText("foreground service running")
                    setOngoing(true)
                    setSmallIcon(R.drawable.ic_launcher_foreground)
                }
                .build(),
        )
    }

    private fun setupChannel() {
        if (notificationManager.getNotificationChannel(CHANNEL_ID) == null) {
            val channel = NotificationChannel("foreground", "ForegroundService", NotificationManager.IMPORTANCE_DEFAULT)
            notificationManager.createNotificationChannel(channel)
        }
    }

    private fun stopForeground() {
        ServiceCompat.stopForeground(this, ServiceCompat.STOP_FOREGROUND_REMOVE)
        notificationManager.cancel(NOTIFICATION_ID)
    }

    override fun onDestroy() {
        super.onDestroy()
        coroutineScope.cancel()
        stopForeground()

        LocalBroadcastManager.getInstance(this).unregisterReceiver(receiver)
    }

    companion object {

        private const val ACTION_STOP = "com.vulpeszerda.navigation.component.sample.ACTION_STOP_FOREGROUND_SERVICE"
        private const val NOTIFICATION_ID = 1
        private const val CHANNEL_ID = "foreground"

        fun start(context: Context) {
            ContextCompat.startForegroundService(context, Intent(context, ForegroundService::class.java))
        }

        fun stop(context: Context) {
            LocalBroadcastManager.getInstance(context).sendBroadcast(Intent(ACTION_STOP))
        }
    }
}
