package org.test.pagingerrorpropogationbug

import android.content.Context
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material3.Button
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.Stable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.paging.Pager
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import androidx.paging.PagingSource
import androidx.paging.compose.collectAsLazyPagingItems
import androidx.paging.compose.itemKey
import androidx.room.Dao
import androidx.room.Database
import androidx.room.Entity
import androidx.room.Fts4
import androidx.room.Query
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.sqlite.db.SupportSQLiteDatabase
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flowOf
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import org.test.pagingerrorpropogationbug.ui.theme.PagingErrorPropogationBugTheme
import java.util.concurrent.Executors

// Component used: Room + Paging 3 (particularly LimitOffsetPagingSource class)
// Version used: Room (2.6.0-rc01), Paging (3.3.0-alpha02)
// devices/Android versions reproduced on: Emulator (API 34)
//
// This sample app showcases how there is no error propagation available when using default PagingSource in queries.
// The sample app has a text field that filters a list of animals. The MATCH statement is used to throw an Exception
// but there may be other ways to do this as well.
//
// Steps to reproduce:
//      1. Build and run the app
//      2. Click the CRASH button or type "OR" into the text field
//      3. Set the PREFLIGHT_ENABLED flag to true, run the app again and the crash will not happen
//
// Under the covers, the default implementation of PagingSource in DAO queries is LimitOffsetPagingSource.
// Exceptions are not propagated on the coroutine scope the query is called on for handling.
// Exceptions are not propagated to the database query executor for handling.
//
// In this particular case, we have options to mitigate this:
//      1. We can preflight all MATCH requests before executing PagingSource DAO queries (demonstrated in this app)
//      2. We can write our own PagingSource that handles exceptions ourselves
//      3. There may be other solutions...
//
// In lieu of an actual fix, there should probably be documentation that acknowledges and warns about situations using
// the default PagingSource, particularly in instances where the MATCH statement is used with user input since this is
// a common use case with FTS4 entities that requires special care.
//
// There are a few StackOverflow issues that mention this issue, both reported years ago, but I could not find any
// bug reports:
//      https://stackoverflow.com/questions/63657792/how-do-we-handle-runtime-errors-from-room-generated-paging-code
//      https://stackoverflow.com/questions/59859531/how-to-catch-unhandled-exceptions-in-the-room-persistence-library?rq=4

private const val PREFLIGHT_ENABLED = false // NOTE: toggle to true to prevent crashes

class MainActivity : ComponentActivity() {
    private lateinit var viewModel: AppViewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        viewModel = AppViewModel(this) // Can only happen after onCreate
        setContent {
            PagingErrorPropogationBugTheme {
                Surface(modifier = Modifier.fillMaxSize(), color = MaterialTheme.colorScheme.background) {
                    val state by viewModel.state.collectAsState()
                    AppContent(state, viewModel::onQueryChange)
                }
            }
        }
    }
}

@Composable
fun AppContent(state: AppState = AppState(), onQueryChange: (TextFieldValue) -> Unit = {}) {
    Column(modifier = Modifier.fillMaxSize()) {
        Row {
            Button(onClick = { onQueryChange(TextFieldValue("a*")) }) { Text("Animals starting with 'A'") }
            Button(onClick = { onQueryChange(TextFieldValue("a* OR ")) }) { Text("CRASH") }
        }
        TextField(
            modifier = Modifier.fillMaxWidth(),
            value = state.query,
            onValueChange = onQueryChange
        )
        val lazyPagingItems = state.data.collectAsLazyPagingItems()
        LazyColumn(modifier = Modifier.fillMaxSize()) {
            items(count = lazyPagingItems.itemCount, key = lazyPagingItems.itemKey { it.name }) { index ->
                val item = lazyPagingItems[index]!!
                Text(modifier = Modifier.padding(10.dp), text = item.name)
            }
        }
    }
}

@Preview(showBackground = true)
@Composable
fun GreetingPreview() {
    PagingErrorPropogationBugTheme {
        AppContent()
    }
}

@Stable
data class AppState(
    val query: TextFieldValue = TextFieldValue(),
    val data: Flow<PagingData<Animal>> = flowOf()
)

class AppViewModel(context: Context) {
    private val coroutineExceptionHandler = CoroutineExceptionHandler { _, throwable ->
        Log.e(null, "Exception caught in coroutine exception handler", throwable)
    }
    private val coroutineScope = CoroutineScope(Dispatchers.IO + coroutineExceptionHandler)
    private val database = Room
//        .databaseBuilder(context, AppDatabase::class.java, "database")
        .inMemoryDatabaseBuilder(context, AppDatabase::class.java)
        .addCallback(callback = object : RoomDatabase.Callback() {
            override fun onCreate(db: SupportSQLiteDatabase) {
                super.onCreate(db)
                Log.d(null, "Database created")
                listOf("Aardvark", "Ape", "Bat", "Bull", "Cat", "Crane").forEach { name ->
                    db.execSQL("INSERT INTO Animal(name) VALUES(?)", arrayOf(name))
                    Log.d(null, "Animal inserted: $name")
                }
            }
        })
        .setQueryExecutor(Executors.newSingleThreadExecutor { runnable ->
            val thread = Thread {
                Log.d(null, "Running query")
                try {
                    runnable.run()
                } catch (e: Exception) {
                    Log.d(null, "Exception caught in thread catch stmt", e)
                }
            }
//            thread.setUncaughtExceptionHandler { t, e ->
//                Log.e(null, "Exception caught in thread exception handler: $t", e)
//            }
            thread
        })
        .build()
    private val dao = database.animalDao()
    private val _state = MutableStateFlow(AppState())
    val state: StateFlow<AppState> = _state

    fun onQueryChange(query: TextFieldValue) {
        coroutineScope.launch {
            Log.d(null, "Query update: ${query.text}")
            if (PREFLIGHT_ENABLED) {
                try {
                    dao.preflightSelectAnimalByMatch(query.text)
                } catch (e: Exception) {
                    Log.w(null, "Exception caught in preflight check", e)
                    _state.update { it.copy(query = query, data = flowOf()) }
                    return@launch
                }
            }
            val animals = dao.selectAnimalByMatch(query.text)
            val data = Pager(PagingConfig(25)) { animals }.flow.catch { cause ->
                Log.e(null, "Exception caught in flow catch stmt", cause)
            }
            _state.update { it.copy(query = query, data = data) }
        }
    }
}

@Fts4
@Entity
data class Animal(val name: String)

@Dao
interface AnimalDao {
    @Query("SELECT * FROM Animal WHERE Animal MATCH :query")
    fun selectAnimalByMatch(query: String): PagingSource<Int, Animal>

    @Query("SELECT 1 FROM Animal WHERE Animal MATCH :query")
    fun preflightSelectAnimalByMatch(query: String): Int
}

@Database(entities = [Animal::class], version = 1)
abstract class AppDatabase : RoomDatabase() {
    abstract fun animalDao(): AnimalDao
}