package com.example.jumpingtransition

import android.graphics.Color
import android.os.Bundle
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.FrameLayout
import android.widget.LinearLayout
import androidx.activity.BackEventCompat
import androidx.activity.OnBackPressedCallback
import androidx.appcompat.app.AppCompatActivity
import androidx.core.view.OneShotPreDrawListener
import androidx.core.view.contains
import androidx.core.view.isVisible
import androidx.transition.Slide
import androidx.transition.TransitionManager
import androidx.transition.TransitionSeekController
import androidx.transition.TransitionSet
import androidx.transition.addListener


class MainActivity : AppCompatActivity() {
    lateinit var container: FrameLayout
    lateinit var blueScreen: FrameLayout
    lateinit var greenScreen: FrameLayout

    lateinit var animateButton: Button
    lateinit var buttonContainer: LinearLayout
    lateinit var cancelButton: Button
    lateinit var completeButton: Button

    var controller: TransitionSeekController? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)

        container = findViewById(R.id.container)
        blueScreen = findViewById(R.id.content1)
        blueScreen.setBackgroundColor(Color.BLUE)
        greenScreen = findViewById(R.id.content2)
        greenScreen.setBackgroundColor(Color.GREEN)

        animateButton = findViewById<Button?>(R.id.animate_button).apply {
            setOnClickListener {
                prepareToAnimate()
                OneShotPreDrawListener.add(container) {
                    if ((controller as TransitionSeekController).isReady) {
                        Log.d("TransitionTest", "Seeking half way")
                        (controller as TransitionSeekController).currentFraction = 0.5f
                    }
                }
            }
        }
        buttonContainer = findViewById(R.id.button_container)
        cancelButton = findViewById<Button?>(R.id.cancel_button).apply {
            setOnClickListener { cancelTransition() }
        }
        completeButton = findViewById<Button>(R.id.complete_button).apply {
            setOnClickListener { completeTransition() }
        }

        buttonContainer.visibility = View.GONE

        // You can swap out these 2 lines to use visibility instead of add/remove
        greenScreen.visibility = View.INVISIBLE
//        container.removeView(greenScreen)

        onBackPressedDispatcher.addCallback(this, object: OnBackPressedCallback(true) {

            override fun handleOnBackStarted(backEvent: BackEventCompat) {
                prepareToAnimate()
            }

            override fun handleOnBackProgressed(backEvent: BackEventCompat) {
                if (controller!!.isReady) {
                    var time: Long = (backEvent.progress * controller!!.durationMillis).toLong()
                    // We cannot let the time get to 0 or the totalDuration to avoid
                    // completing the operation accidentally.
                    if (time == 0L) {
                        time = 1L
                    }
                    if (time == controller!!.durationMillis) {
                        time = controller!!.durationMillis - 1
                    }
                    controller!!.currentPlayTimeMillis = time
                }
            }

            override fun handleOnBackPressed() {
                completeTransition()
            }

            override fun handleOnBackCancelled() {
                cancelTransition()
            }
        })
    }

    private fun prepareToAnimate() {
        Log.d("TransitionTest", "Blue screen:  parent = ${blueScreen.parent} " +
                "visibility = ${blueScreen.visibility}")
        Log.d("TransitionTest", "Green screen:  parent = ${greenScreen.parent} " +
                "visibility = ${greenScreen.visibility}")
        val transitionIn = Slide(Gravity.LEFT).apply {
            duration = 2000
            addTarget(blueScreen)
        }
        val transitionOut = Slide(Gravity.RIGHT).apply {
            duration = 2000
            addTarget(greenScreen)
        }

        val transition = TransitionSet().apply {
            addTransition(transitionIn)
            addTransition(transitionOut)
        }

        transition.addListener(onEnd = {
            Log.d("TransitionTest", "onTransitionEnd of transition listener")
        })

        controller = TransitionManager.controlDelayedTransition(container, transition)
        reverseViews()

        animateButton.visibility = View.GONE
        buttonContainer.visibility = View.VISIBLE
    }

    private fun cancelTransition() {
        Log.d("TransitionTest", "Animating to Start")
        controller?.animateToStart {
            Log.d("TransitionTest", "Animate to Start Runnable Running")
            reverseViews()
        }

        animateButton.visibility = View.VISIBLE
        buttonContainer.visibility = View.GONE
    }

    private fun completeTransition() {
        Log.d("TransitionTest", "Animating to End")
        controller?.animateToEnd()
        animateButton.visibility = View.VISIBLE
        buttonContainer.visibility = View.GONE
    }

    private fun reverseViews(useVisibility: Boolean = true) {
        if (useVisibility) {
            if (blueScreen.isVisible) {
                blueScreen.visibility = View.INVISIBLE
                greenScreen.visibility = View.VISIBLE
            } else {
                blueScreen.visibility = View.VISIBLE
                greenScreen.visibility = View.INVISIBLE
            }
        } else {
            if (container.contains(blueScreen)) {
                container.removeView(blueScreen)
                if (greenScreen.parent != null) {
                    (greenScreen.parent as ViewGroup).removeView(greenScreen)
                }
                container.addView(greenScreen)
            } else {
                container.removeView(greenScreen)
                if (blueScreen.parent != null) {
                    (blueScreen.parent as ViewGroup).removeView(blueScreen)
                }
                container.addView(blueScreen)
            }
        }
    }
}