package camerax.reproducer

import android.Manifest
import android.animation.ObjectAnimator
import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.hardware.camera2.CameraCharacteristics
import android.location.Location
import android.os.Build
import android.os.Bundle
import android.os.SystemClock
import android.os.VibrationEffect
import android.os.Vibrator
import android.os.VibratorManager
import android.util.Log
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.OrientationEventListener
import android.view.ScaleGestureDetector
import android.view.Surface
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.widget.Toast
import androidx.camera.camera2.interop.Camera2CameraInfo
import androidx.camera.core.AspectRatio
import androidx.camera.core.Camera
import androidx.camera.core.CameraInfo
import androidx.camera.core.CameraSelector
import androidx.camera.core.CameraState
import androidx.camera.core.FocusMeteringAction
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCaptureException
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.camera.video.FallbackStrategy
import androidx.camera.video.FileOutputOptions
import androidx.camera.video.Quality
import androidx.camera.video.QualitySelector
import androidx.camera.video.Recorder
import androidx.camera.video.Recording
import androidx.camera.video.VideoCapture
import androidx.camera.video.VideoRecordEvent
import androidx.core.animation.doOnEnd
import androidx.core.animation.doOnStart
import androidx.core.content.ContextCompat
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.flowWithLifecycle
import androidx.lifecycle.lifecycleScope
import camerax.reproducer.databinding.CameraFragmentPreviewBinding
import camerax.reproducer.databinding.CameraFragmentUiBinding
import coil.ImageLoader
import coil.disk.DiskCache
import coil.memory.MemoryCache
import coil.request.CachePolicy
import coil.request.ImageRequest
import coil.transform.RoundedCornersTransformation
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import java.io.File
import java.time.ZonedDateTime
import java.time.format.DateTimeFormatter
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min


@AndroidEntryPoint
@Suppress("LargeClass")
class CameraFragment : Fragment() {

    companion object {
        private const val RATIO_4_3_VALUE = 4.0 / 3.0
        private const val RATIO_16_9_VALUE = 16.0 / 9.0
        private const val CAPTURE_FAILURE_MESSAGE = "Failed to capture photo."
        private const val ROTATION_DURATION = 500L
        private const val TURN = 90F
        private const val VIBRATION_MS = 8L
        private const val COVER_START_OPACITY = .8F
        private const val COVER_ANMATION_DURATION = 300L
        const val FILENAME_FORMAT = "uuuuMMdd-HHmmss_S"
        const val FILE_UPLOAD_SIZE_LIMIT = 1000L * 1024L * 1024L
        const val CORNER_RADIUS = 8F
    }

    val viewModel: CameraFragmentViewModel by viewModels()

    private var _fragmentCameraBinding: CameraFragmentPreviewBinding? = null
    private val fragmentCameraBinding get() = _fragmentCameraBinding!!
    private var cameraUiContainerBinding: CameraFragmentUiBinding? = null

    private var displayId: Int = -1
    private var lensFacing: Int = CameraSelector.LENS_FACING_BACK
    private var preview: Preview? = null
    private var imageCapture: ImageCapture? = null
    private var videoCapture: VideoCapture<Recorder>? = null
    private var recording: Recording? = null
    private var camera: Camera? = null
    private var cameraProvider: ProcessCameraProvider? = null

    private var vibrator: Vibrator? = null
    private val vibration = VibrationEffect.createOneShot(VIBRATION_MS, VibrationEffect.DEFAULT_AMPLITUDE)


    private lateinit var windowManager: WindowManager

    /** Blocking camera operations are performed using this executor */
    private lateinit var cameraExecutor: ExecutorService

    private lateinit var orientationEventListener: OrientationEventListener

    private lateinit var imageLoader: ImageLoader

    private var finishOnStop = false

    private lateinit var mediaDirectory: File

    private val scaleGestureListener = object : ScaleGestureDetector.SimpleOnScaleGestureListener() {
        override fun onScale(detector: ScaleGestureDetector): Boolean {
            // Get the camera's current zoom ratio
            val currentZoomRatio = camera?.cameraInfo?.zoomState?.value?.zoomRatio ?: 0F

            // Get the pinch gesture's scaling factor
            val delta = detector.scaleFactor

            // Update the camera's zoom ratio. This is an asynchronous operation that returns
            // a ListenableFuture, allowing you to listen to when the operation completes.
            camera?.cameraControl?.setZoomRatio(currentZoomRatio * delta)

            // Return true, as the event was handled
            return true
        }
    }


    override fun onDestroyView() {
        _fragmentCameraBinding = null
        orientationEventListener.disable()
        super.onDestroyView()

        // Shut down our background executor
        cameraExecutor.shutdown()
    }

    private var previousOrientation: Int = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        // Initialize our background executor.
        cameraExecutor = Executors.newCachedThreadPool()
    }

    override fun onAttach(context: Context) {
        super.onAttach(context)
        mediaDirectory = File(context.filesDir, "media")
        val coilCacheDirectory = File(context.cacheDir, "coil")
        lifecycleScope.launch {
            if (!mediaDirectory.exists()) {
                mediaDirectory.mkdirs()
            }
            if (!coilCacheDirectory.exists()) {
                coilCacheDirectory.mkdirs()
            }
        }

        imageLoader = ImageLoader.Builder(context)
            .memoryCachePolicy(CachePolicy.ENABLED)
            .diskCachePolicy(CachePolicy.ENABLED)
            .networkCachePolicy(CachePolicy.ENABLED)
            // for now, ignore cache.
            .respectCacheHeaders(false)
            .memoryCache {
                MemoryCache.Builder(context)
                    .maxSizePercent(0.25)
                    .build()
            }
            .diskCache {
                DiskCache.Builder()
                    .directory(coilCacheDirectory)
                    .maxSizeBytes(50L * 1024L * 1024L)
                    .build()
            }
            .build()

    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        vibrator = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            ContextCompat.getSystemService(requireContext(), VibratorManager::class.java)?.defaultVibrator
        } else {
            ContextCompat.getSystemService(requireContext(), Vibrator::class.java)
        }

        // reset orientation just in case.
        previousOrientation = 0

        // main view binding.
        _fragmentCameraBinding = CameraFragmentPreviewBinding.inflate(inflater, container, false)

        @Suppress("MagicNumber")
        with(fragmentCameraBinding.chronometer) {
            format = "00:%s"
            setOnChronometerTickListener { cArg ->
                val elapsedMillis = SystemClock.elapsedRealtime() - cArg.base
                if (elapsedMillis > 3600000L) {
                    cArg.format = "0%s"
                } else {
                    cArg.format = "00:%s"
                }
            }
        }

        // orientation listener.
        // feed it into a flow to smooth out the values.
        orientationEventListener = object : OrientationEventListener(requireContext()) {
            override fun onOrientationChanged(orientation: Int) {
                viewModel.setOrientation(orientation)
            }
        }.also {
            it.enable()
        }

        // get the orientation
        viewModel.orientation
            .flowWithLifecycle(viewLifecycleOwner.lifecycle)
            .distinctUntilChanged()
            .onEach { newOrientation ->
                // set the camera target orientation.
                imageCapture?.targetRotation = newOrientation
                @SuppressLint("RestrictedApi")
                videoCapture?.targetRotation = newOrientation

                // if the orientation has changed, we need to rotate our toolbar items
                if (previousOrientation != newOrientation) {
                    val rotations = mutableListOf<ObjectAnimator>()

                    _fragmentCameraBinding?.flashIcon?.let {
                        ObjectAnimator.ofFloat(it, "rotation", previousOrientation * TURN, newOrientation * TURN).apply {
                            duration = ROTATION_DURATION
                        }
                    }?.let {
                        rotations.add(it)
                    }

                    _fragmentCameraBinding?.backButton?.let {
                        ObjectAnimator.ofFloat(it, "rotation", previousOrientation * TURN, newOrientation * TURN).apply {
                            duration = ROTATION_DURATION
                        }
                    }?.let {
                        rotations.add(it)
                    }

                    _fragmentCameraBinding?.root?.post {
                        rotations.forEach {
                            it.start()
                        }
                    }

                    previousOrientation = newOrientation
                }
            }
            .launchIn(viewLifecycleOwner.lifecycleScope)

        (requireActivity() as? CameraActivity)?.let {
            it.hideSystemUI()
            updateToolbar()
        }

        return fragmentCameraBinding.root
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        setupLastCapture()

        //Initialize WindowManager to retrieve display metrics
        windowManager = requireActivity().windowManager

        // Wait for the views to be properly laid out
        fragmentCameraBinding.viewFinder.post {

            // Keep track of the display in which this view is attached
            displayId = fragmentCameraBinding.viewFinder.display.displayId

            // Build UI controls
            updateCameraUi()

            // Set up the camera and its use cases
            setUpCamera()
        }

        val scaleGestureDetector = ScaleGestureDetector(requireContext(), scaleGestureListener)

        fragmentCameraBinding.viewFinder.setOnTouchListener { _: View, motionEvent: MotionEvent ->
            when (motionEvent.action) {
                MotionEvent.ACTION_DOWN -> return@setOnTouchListener true
                MotionEvent.ACTION_UP -> {
                    // Get the MeteringPointFactory from PreviewView
                    val factory = fragmentCameraBinding.viewFinder.meteringPointFactory

                    // Create a MeteringPoint from the tap coordinates
                    val point = factory.createPoint(motionEvent.x, motionEvent.y)

                    // Create a MeteringAction from the MeteringPoint, you can configure it to specify the metering mode
                    val action = FocusMeteringAction.Builder(point).build()

                    // Trigger the focus and metering. The method returns a ListenableFuture since the operation
                    // is asynchronous. You can use it get notified when the focus is successful or if it fails.
                    camera?.cameraControl?.startFocusAndMetering(action)

                    return@setOnTouchListener true
                }

                else -> return@setOnTouchListener scaleGestureDetector.onTouchEvent(motionEvent)
            }
        }


        viewModel.captureMode
            .flowWithLifecycle(viewLifecycleOwner.lifecycle)
            .distinctUntilChanged()
            .onEach {
                cameraUiContainerBinding?.captureButton?.setImageDrawable(
                    ContextCompat.getDrawable(
                        requireContext(),
                        when (it) {
                            CameraCaptureMode.PHOTO -> {
                                R.drawable.capture_image
                            }

                            CameraCaptureMode.VIDEO -> {
                                R.drawable.capture_video
                            }
                        }
                    )
                )

                // when we switch modes, reset the flash mode.
                // this way we shouldn't see auto focus when in video mode.
                viewModel.setFlashMode(ImageCapture.FLASH_MODE_OFF)
            }
            .launchIn(viewLifecycleOwner.lifecycleScope)

        fragmentCameraBinding.backButton.setOnClickListener {
            shutdown()
        }

        setupFlash()
    }

    private fun shutdown() {
        recording?.let {
            finishOnStop = true
            stopRecording()
        } ?: run {
            requireActivity().finish()
        }
    }

    private fun setupLastCapture() {
        viewModel.lastCapturePath
            .flowWithLifecycle(viewLifecycleOwner.lifecycle)
            .onEach { path ->
                cameraUiContainerBinding?.let { binding ->
                    if (path == null) {
                        binding.lastCapture.setImageDrawable(null)
                        binding.lastCapture.visibility = View.INVISIBLE
                        binding.doneButton.isEnabled = false
                    } else {
                        binding.doneButton.isEnabled = true
                        val request = ImageRequest.Builder(requireContext())
                            .data(path)
                            .target(binding.lastCapture)
                            .transformations(
                                listOf(
                                    RoundedCornersTransformation(
                                        CORNER_RADIUS,
                                        CORNER_RADIUS,
                                        CORNER_RADIUS,
                                        CORNER_RADIUS
                                    )
                                )
                            )
                         .build()

                        imageLoader.enqueue(request)
                    }
                }
            }.launchIn(viewLifecycleOwner.lifecycleScope)
    }

    private fun updateToolbar() {
        // This will bump the toolbar down so that it doesn't overlap the
        // camera cutout, if there is one on the device.
        fragmentCameraBinding.root.addOnLayoutChangeListener { _, _, _, _, _, _, _, _, _ ->
            fragmentCameraBinding.root.rootWindowInsets.displayCutout?.safeInsetTop?.let { safeInsetTop ->
                (_fragmentCameraBinding?.toolbar?.layoutParams as? ViewGroup.MarginLayoutParams)?.let { newLayoutParams ->
                    if (newLayoutParams.topMargin != safeInsetTop) {
                        newLayoutParams.setMargins(0, safeInsetTop, 0, 0)
                        _fragmentCameraBinding?.toolbar?.layoutParams = newLayoutParams
                    }
                }
            }
        }
    }

    override fun onPause() {
        stopRecording()
        super.onPause()
    }

    /**
     * Inflate camera controls and update the UI manually upon config changes to avoid removing
     * and re-adding the view finder from the view hierarchy; this provides a seamless rotation
     * transition on devices that support it.
     *
     * NOTE: The flag is supported starting in Android 8 but there still is a small flash on the
     * screen for devices that run Android 9 or below.
     */
    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)

        // Rebind the camera with the updated display metrics
        bindCameraUseCases()
    }

    /** Initialize CameraX, and prepare to bind the camera use cases  */
    private fun setUpCamera() {
        if (_fragmentCameraBinding == null) {
            return
        }
        val cameraProviderFuture = ProcessCameraProvider.getInstance(requireContext())
        cameraProviderFuture.addListener({

            // CameraProvider
            cameraProvider = cameraProviderFuture.get()

            // Select lensFacing depending on the available cameras
            lensFacing = when {
                hasBackCamera() -> CameraSelector.LENS_FACING_BACK
                hasFrontCamera() -> CameraSelector.LENS_FACING_FRONT
                else -> error("Back and front camera are unavailable")
            }

            // Build and bind the camera use cases
            bindCameraUseCases()
        }, ContextCompat.getMainExecutor(requireContext()))
    }

    /** Declare and bind preview, capture and analysis use cases */
    private fun bindCameraUseCases() {

        // if we have a camera, and it has a flash, ensure we turn the flash off.
        if (camera?.cameraInfo?.hasFlashUnit() == true) {
            camera?.cameraControl?.enableTorch(false)
        }

        if (_fragmentCameraBinding == null) {
            return
        }

        // Get screen metrics used to setup camera for full screen resolution
        val screenAspectRatio = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            with(windowManager.currentWindowMetrics.bounds) {
                aspectRatio(this.width(), this.height())
            }
        } else {
            with(resources.displayMetrics) {
                aspectRatio(this.widthPixels, this.heightPixels)
            }
        }

        // CameraProvider
        val cameraProvider = cameraProvider ?: error("Camera initialization failed.")

        // CameraSelector
        val cameraSelector = CameraSelector.Builder().requireLensFacing(lensFacing).build()

        // Preview
        preview = Preview.Builder()
            // We request aspect ratio but no resolution
            .setTargetAspectRatio(screenAspectRatio)
            // Set initial target rotation
            .setTargetRotation(Surface.ROTATION_0)
            .build()

        @androidx.camera.camera2.interop.ExperimentalCamera2Interop
        val cameraInfo = cameraProvider.availableCameraInfos.filter {
            Camera2CameraInfo.from(it).getCameraCharacteristic(CameraCharacteristics.LENS_FACING) == lensFacing
        }

        val supportedQualities = QualitySelector.getSupportedQualities(cameraInfo[0])
        // removed Quality.UHD for now - files are big, streaming is slow
        val filteredQualities = arrayListOf(Quality.FHD, Quality.HD, Quality.SD)
            .filter { supportedQualities.contains(it) }

        val qualitySelector = QualitySelector.fromOrderedList(
            filteredQualities,
            FallbackStrategy.lowerQualityOrHigherThan(Quality.SD)
        )

        val recorder = Recorder.Builder()
            .setQualitySelector(qualitySelector)
            .build()

        @SuppressLint("RestrictedApi")
        videoCapture = VideoCapture.withOutput(recorder).apply {
            setTargetRotation((_fragmentCameraBinding?.viewFinder?.display?.rotation ?: viewModel.orientation.value))
        }

        // ImageCapture
        imageCapture = ImageCapture.Builder()
            .setCaptureMode(ImageCapture.CAPTURE_MODE_MINIMIZE_LATENCY)
            .apply {
                if (camera?.cameraInfo?.hasFlashUnit() == true) {
                    setFlashMode(viewModel.flashStatus.value)
                }
            }
            // We request aspect ratio but no resolution to match preview config, but letting
            // CameraX optimize for whatever specific resolution best fits our use cases
            .setTargetAspectRatio(screenAspectRatio)
            // Set initial target rotation, we will have to call this again if rotation changes
            // during the lifecycle of this use case
            // Set initial target rotation
            .setTargetRotation(
                (_fragmentCameraBinding?.viewFinder?.display?.rotation ?: viewModel.orientation.value)
            )
            .build()

        // Must unbind the use-cases before rebinding them
        cameraProvider.unbindAll()

        if (camera != null) {
            // Must remove observers from the previous camera instance
            removeCameraStateObservers(camera!!.cameraInfo)
        }

        try {
            // the preview surface needs to be set before binding to the lifecycel when videoCapture is included.
            preview?.setSurfaceProvider(fragmentCameraBinding.viewFinder.surfaceProvider)

            camera = cameraProvider.bindToLifecycle(this, cameraSelector, imageCapture, videoCapture, preview)

            // update flash status based on camera.
            _fragmentCameraBinding?.flashIcon?.isEnabled = camera?.cameraInfo?.hasFlashUnit() == true

            // Attach the viewfinder's surface provider to preview use case
            observeCameraState(camera?.cameraInfo!!)
        } catch (@Suppress("TooGenericExceptionCaught") e: Exception) {
            Log.e("CAMERA","Use case binding failed",e)
        }
    }

    private fun setupFlash() {
        // update the ui to match the status and reconfigure the camera.
        viewModel.flashStatus.flowWithLifecycle(viewLifecycleOwner.lifecycle).distinctUntilChanged().onEach { status ->
            fragmentCameraBinding.flashIcon.setFlashMode(status)
            if (camera != null && recording == null) {
                bindCameraUseCases()
            }
        }.launchIn(viewLifecycleOwner.lifecycleScope)

        fragmentCameraBinding.flashIcon.setOnClickListener {
            if (camera?.cameraInfo?.hasFlashUnit() == true) {
                if (viewModel.captureMode.value == CameraCaptureMode.PHOTO && recording == null) {
                    viewModel.toggleFlash()
                } else {
                    when (fragmentCameraBinding.flashIcon.getFlashMode()) {
                        ImageCapture.FLASH_MODE_OFF -> {
                            viewModel.setFlashMode(ImageCapture.FLASH_MODE_ON)
                            camera?.cameraControl?.enableTorch(true)
                        }

                        ImageCapture.FLASH_MODE_AUTO -> {
                            viewModel.setFlashMode(ImageCapture.FLASH_MODE_OFF)
                            camera?.cameraControl?.enableTorch(false)
                        }

                        ImageCapture.FLASH_MODE_ON -> {
                            camera?.cameraControl?.enableTorch(false)
                            viewModel.setFlashMode(ImageCapture.FLASH_MODE_OFF)
                        }
                    }
                }
            }
        }
    }

    private fun removeCameraStateObservers(cameraInfo: CameraInfo) {
        cameraInfo.cameraState.removeObservers(viewLifecycleOwner)
    }

    @Suppress("CyclomaticComplexMethod")
    private fun observeCameraState(cameraInfo: CameraInfo) {
        cameraInfo.cameraState.observe(viewLifecycleOwner) { cameraState ->
            run {
                when (cameraState.type) {
                    CameraState.Type.PENDING_OPEN -> {
                        Log.w("CAMERA","Camera Pending Open. Advise user to close other camera apps?")
                    }

                    CameraState.Type.OPENING -> {
                        Log.w("CAMERA","Camera is opening.")
                    }

                    CameraState.Type.OPEN -> {
                        Log.w("CAMERA","Camera is open.")
                    }

                    CameraState.Type.CLOSING -> {
                        Log.w("CAMERA","Camera is closing.")
                    }

                    CameraState.Type.CLOSED -> {
                        Log.w("CAMERA", "Camera is closed.")
                    }
                }
            }

            cameraState.error?.let { error ->
                when (error.code) {
                    // Open errors
                    CameraState.ERROR_STREAM_CONFIG -> {
                       "Stream config error: ${error.code} ${error.type}"
                    }
                    // Opening errors
                    CameraState.ERROR_CAMERA_IN_USE -> {
                        "Camera in use:  ${error.code} ${error.type}"
                    }

                    CameraState.ERROR_MAX_CAMERAS_IN_USE -> {
                        "Max cameras in use: ${error.code} ${error.type}"
                        // Close another open camera in the app, or ask the user to close another
                        // camera app that's using the camera
                    }

                    CameraState.ERROR_OTHER_RECOVERABLE_ERROR -> {
                        "Other recoverable error: ${error.code} ${error.type}"
                    }
                    // Closing errors
                    CameraState.ERROR_CAMERA_DISABLED -> {
                        // Ask the user to enable the device's cameras
                        "Camera disabled: ${error.code} ${error.type}"
                    }

                    CameraState.ERROR_CAMERA_FATAL_ERROR -> {
                        "Fatal error: ${error.code} ${error.type}"
                    }
                    // Closed errors
                    CameraState.ERROR_DO_NOT_DISTURB_MODE_ENABLED -> {
                        // Ask the user to disable the "Do Not Disturb" mode, then reopen the camera
                        "Do not disturb is on: ${error.code} ${error.type}"
                    }

                    else -> { "Camera Error ${error.code} occurred." }

                }.let {
                    Log.e("CAMERA", it, error.cause)
                }
            }
        }
    }

    /**
     *  [androidx.camera.core.ImageAnalysis.Builder] requires enum value of
     *  [androidx.camera.core.AspectRatio]. Currently it has values of 4:3 & 16:9.
     *
     *  Detecting the most suitable ratio for dimensions provided in @params by counting absolute
     *  of preview ratio to one of the provided values.
     *
     *  @param width - preview width
     *  @param height - preview height
     *  @return suitable aspect ratio
     */
    private fun aspectRatio(width: Int, height: Int): Int {
        val previewRatio = max(width, height).toDouble() / min(width, height)
        if (abs(previewRatio - RATIO_4_3_VALUE) <= abs(previewRatio - RATIO_16_9_VALUE)) {
            return AspectRatio.RATIO_4_3
        }
        return AspectRatio.RATIO_16_9
    }

    /** Method used to re-draw the camera UI controls, called every time configuration changes. */
    @SuppressLint("MissingPermission")
    private fun updateCameraUi() {

        // Remove previous UI if any
        cameraUiContainerBinding?.root?.let {
            fragmentCameraBinding.root.removeView(it)
        }

        cameraUiContainerBinding = CameraFragmentUiBinding.inflate(
            LayoutInflater.from(requireContext()),
            fragmentCameraBinding.root,
            true
        )

        cameraUiContainerBinding?.doneButton?.setOnClickListener {
            shutdown()
        }

        cameraUiContainerBinding?.captureButton?.setImageDrawable(
            ContextCompat.getDrawable(
                requireContext(),
                when (viewModel.captureMode.value) {
                    CameraCaptureMode.PHOTO -> {
                        R.drawable.capture_image
                    }

                    CameraCaptureMode.VIDEO -> {
                        R.drawable.capture_video
                    }
                }
            )
        )

        cameraUiContainerBinding?.cameraModeToggle?.addOnButtonCheckedListener { _, checkedId, isChecked ->
            if (isChecked) {
                when (checkedId) {
                    R.id.photo_button -> viewModel.setCaptureMode(CameraCaptureMode.PHOTO)
                    R.id.video_button -> viewModel.setCaptureMode(CameraCaptureMode.VIDEO)
                    else -> Unit
                }
            }
        }
        // Listener for button used to capture photo
        cameraUiContainerBinding?.captureButton?.setOnClickListener {
            when (viewModel.captureMode.value) {
                CameraCaptureMode.PHOTO -> {
                    viewLifecycleOwner.lifecycleScope.launch {


                        takePicture(null)
                    }
                }

                CameraCaptureMode.VIDEO -> if (recording == null) {
                    startRecording()
                } else {
                    stopRecording()
                }
            }

        }
    }

    private fun takePicture(location: Location? = null) {

        // Get a stable reference of the modifiable image capture use case
        imageCapture?.let { imageCapture ->

            // Create output file to hold the image

            // Setup image capture metadata
            val metadata = ImageCapture.Metadata().apply {
                this.location = location
                // Mirror image when using the front camera
                isReversedHorizontal = lensFacing == CameraSelector.LENS_FACING_FRONT
            }

            val outputFile = File(mediaDirectory, DateTimeFormatter.ofPattern(FILENAME_FORMAT).format(
                ZonedDateTime.now()) + ".jpg")

            takePicture(imageCapture, metadata, outputFile)

        }
    }

    private fun takePicture(imageCapture: ImageCapture, metadata: ImageCapture.Metadata, outputFile: File) {
        val outputOptions = ImageCapture.OutputFileOptions.Builder(outputFile)
            .setMetadata(metadata)
            .build()

        val alphaAnimation = ObjectAnimator.ofFloat(fragmentCameraBinding.cover, View.ALPHA, COVER_START_OPACITY, 0F)
            .setDuration(COVER_ANMATION_DURATION).apply {
                doOnStart {
                    fragmentCameraBinding.cover.alpha = COVER_START_OPACITY
                    fragmentCameraBinding.cover.visibility = View.VISIBLE
                }
                doOnEnd {
                    fragmentCameraBinding.cover.visibility = View.GONE
                    fragmentCameraBinding.cover.alpha = COVER_START_OPACITY
                }
            }

        imageCapture.takePicture(
            outputOptions, cameraExecutor, object : ImageCapture.OnImageSavedCallback {
                override fun onCaptureStarted() {
                    super.onCaptureStarted()

                    alphaAnimation.start()
                    vibrator?.cancel()
                    vibrator?.vibrate(vibration)

                }
                override fun onError(e: ImageCaptureException) {
                    Log.e("CAMERA", CAPTURE_FAILURE_MESSAGE, e)
                    // clear manual focus
                    camera?.cameraControl?.cancelFocusAndMetering()
                }

                override fun onImageSaved(output: ImageCapture.OutputFileResults) {
                    // clear manual focus
                    camera?.cameraControl?.cancelFocusAndMetering()
                    // we'd normally call the view model here to save a reference to our image.
                    viewModel.setLastCapturePath(outputFile.absolutePath)
                }
            }
        )
    }


    /** Returns true if the device has an available back camera. False otherwise */
    private fun hasBackCamera(): Boolean {
        return cameraProvider?.hasCamera(CameraSelector.DEFAULT_BACK_CAMERA) ?: false
    }

    /** Returns true if the device has an available front camera. False otherwise */
    private fun hasFrontCamera(): Boolean {
        return cameraProvider?.hasCamera(CameraSelector.DEFAULT_FRONT_CAMERA) ?: false
    }

    private fun startRecording() {
        val outputFile = File(mediaDirectory, DateTimeFormatter.ofPattern(FILENAME_FORMAT).format(
            ZonedDateTime.now()) + ".mp4")
        val fileOutputOptions = FileOutputOptions.Builder(outputFile)
            .setFileSizeLimit(FILE_UPLOAD_SIZE_LIMIT)
            .build()

        recording = videoCapture?.output?.prepareRecording(requireContext(), fileOutputOptions)?.apply {
            if (ContextCompat.checkSelfPermission(requireContext(), Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED) {
                withAudioEnabled()
            }
        }?.start(ContextCompat.getMainExecutor(requireContext())) { recordEvent ->
            when (recordEvent) {
                is VideoRecordEvent.Start -> {
                    fragmentCameraBinding.chronometer.base = SystemClock.elapsedRealtime()
                    fragmentCameraBinding.chronometer.start()
                    fragmentCameraBinding.recordingTimer.visibility = View.VISIBLE
                    cameraUiContainerBinding?.let {
                        // hide the done button
                        it.doneButton.visibility = View.GONE

                        // disable mode toggle
                        it.cameraModeToggle.visibility = View.INVISIBLE
                        it.cameraModeToggle.isEnabled = false
                        it.photoButton.isEnabled = false
                        it.videoButton.isEnabled = false

                        // change capture button to stop button
                        it.captureButton.setImageDrawable(ContextCompat.getDrawable(requireContext(), R.drawable.capture_video_stop))
                    }
                }

                is VideoRecordEvent.Finalize -> {

                    var saveFile = true
                    var continueRecording = false

                    if (!recordEvent.hasError()) {
                        Log.d("CAMERA","Video recording ended without error.")
                    } else {

                        when (recordEvent.error) {
                            VideoRecordEvent.Finalize.ERROR_NONE -> Unit

                            // these errors are recoverable, we can proceed as normal,
                            VideoRecordEvent.Finalize.ERROR_FILE_SIZE_LIMIT_REACHED,
                            VideoRecordEvent.Finalize.ERROR_DURATION_LIMIT_REACHED,
                            -> {
                                continueRecording = true
                            }

                            // these we can't recover from, and should abandon the output.
                            VideoRecordEvent.Finalize.ERROR_INSUFFICIENT_STORAGE -> {
                                Toast.makeText(
                                    requireContext(),
                                    "Insufficient disk space for recording. Please free up space and try again.",
                                    Toast.LENGTH_LONG
                                ).show()
                                // don't save file, alert the user.
                                saveFile = false
                            }

                            VideoRecordEvent.Finalize.ERROR_UNKNOWN,
                            VideoRecordEvent.Finalize.ERROR_ENCODING_FAILED,
                            VideoRecordEvent.Finalize.ERROR_INVALID_OUTPUT_OPTIONS,
                            VideoRecordEvent.Finalize.ERROR_RECORDER_ERROR,
                            VideoRecordEvent.Finalize.ERROR_SOURCE_INACTIVE,
                            VideoRecordEvent.Finalize.ERROR_NO_VALID_DATA,
                            -> {
                                recordEvent.cause?.let {
                                    Log.e("CAMERA", "Recording Video Failed. Code: ${recordEvent.error}", it)
                                }
                                Toast.makeText(
                                    requireContext(),
                                    "Recording failed.",
                                    Toast.LENGTH_LONG
                                ).show()
                                saveFile = false
                            }

                            else -> {
                                saveFile = false
                            }
                        }
                    }

                    recording?.close()
                    recording = null

                    if (saveFile) {
                        // we would save a reference to the outputFile.path here.
                    } else {
                        if (outputFile.exists()) {
                            outputFile.delete().takeIf { !it }?.let {
                                Log.d("CAMERA", "Deletion Failed. ${outputFile.absolutePath}")
                            }
                        }
                    }

                    if (finishOnStop) {
                        requireActivity().finish()
                    } else {
                        if (saveFile && outputFile.exists()) {
                            viewModel.setLastCapturePath(outputFile.absolutePath)
                        }

                        fragmentCameraBinding.recordingTimer.visibility = View.GONE
                        fragmentCameraBinding.chronometer.stop()

                        cameraUiContainerBinding?.let {
                            // turn stop button into recording button
                            it.captureButton.setImageDrawable(ContextCompat.getDrawable(requireContext(), R.drawable.capture_video))
                            // re-enable the mode toggle
                            it.cameraModeToggle.visibility = View.VISIBLE
                            it.cameraModeToggle.isEnabled = true
                            it.photoButton.isEnabled = true
                            it.videoButton.isEnabled = true
                            // make the done button visible
                            it.doneButton.visibility = View.VISIBLE
                        }
                        if (continueRecording) {
                            startRecording()
                        }
                    }
                }
            }
        }
    }

    private fun stopRecording() {
        recording?.stop()
        recording = null
    }
}
