package camerax.reproducer

import androidx.lifecycle.ViewModel
import dagger.hilt.android.lifecycle.HiltViewModel
import android.view.Surface
import androidx.camera.core.ImageCapture
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import javax.inject.Inject

@HiltViewModel
class CameraFragmentViewModel @Inject constructor() : ViewModel() {

    private val _captureMode = MutableStateFlow(CameraCaptureMode.PHOTO)
    val captureMode = _captureMode.asStateFlow()

    private val _lastCapturePath = MutableStateFlow<String?>(null)
    val lastCapturePath = _lastCapturePath.asStateFlow()

    private val _orientation = MutableStateFlow(Surface.ROTATION_0)
    val orientation = _orientation.asStateFlow()

    private val _flashStatus = MutableStateFlow(ImageCapture.FLASH_MODE_OFF)
    val flashStatus = _flashStatus.asStateFlow()

    fun setLastCapturePath(absolutePath: String?) {
        _lastCapturePath.value = absolutePath
    }

    @Suppress("MagicNumber")
    fun setOrientation(value: Int) {
        _orientation.value = with(value) {
            when (360 - this) {
                in 330..359, in 0..29 -> Surface.ROTATION_0
                in 60..119 -> Surface.ROTATION_90
                in 150..209 -> Surface.ROTATION_180
                in 240..299 -> Surface.ROTATION_270
                else -> Surface.ROTATION_0
            }
        }
    }

    fun toggleFlash() {
        _flashStatus.value = when (_flashStatus.value) {
            ImageCapture.FLASH_MODE_OFF -> {
                ImageCapture.FLASH_MODE_AUTO
            }

            ImageCapture.FLASH_MODE_AUTO -> {
                ImageCapture.FLASH_MODE_ON
            }

            ImageCapture.FLASH_MODE_ON -> {
                ImageCapture.FLASH_MODE_OFF
            }

            else -> ImageCapture.FLASH_MODE_OFF
        }
    }

    fun setFlashMode(flashMode: Int) {
        _flashStatus.value = flashMode
    }

    fun setCaptureMode(cameraCaptureMode: CameraCaptureMode) {
        _captureMode.value = cameraCaptureMode
    }
}
