package com.example.paging3

import android.app.Application
import android.util.Log
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import androidx.paging.Pager
import androidx.paging.cachedIn
import androidx.paging.insertSeparators
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.util.Calendar
import java.util.Date
import java.util.Locale

class NoteViewModel(application: Application) : AndroidViewModel(application) {

    private var pagingSourceInvalidator: InvalidatingPagingSource.Invalidator? = null
    private val db = NoteDataBase.getDatabase(application)
    private val repository = NoteRepository(db.getNoteDao())
    private val prefs = Preferences(application)
    private val isDataCreated = prefs.getBooleanValue("isDataCreated")

    init {
        if (!isDataCreated) {
            createData(5000)
        }
    }

    val adapterItems = Pager(config = PagingSourceConfig.getConfig()) {
        NotePagingSource(repository, db) {
            pagingSourceInvalidator = it
        }
    }.flow
        .map { pagingData ->


            pagingData.insertSeparators { before: AdapterItem?, after: AdapterItem? ->

                if (before == null && after == null) {
                    // List is empty; return null to skip adding separator.
                    null
                } else if (after == null) {
                    // Footer; return null here to skip adding a footer. For example Load More view.
                    null
                } else if (before == null) {
                    // Header of the list
                    if (after is AdapterItem.Content) {
                        val section = getSection(after.note)
                        AdapterItem.Header(section)
                    } else {
                        null
                    }

                } else {

                    if (before is AdapterItem.Content && after is AdapterItem.Content) {
                        val sectionOfBefore = getSection(before.note)
                        val sectionOfAfter = getSection(after.note)
                        if (sectionOfBefore.headerId != sectionOfAfter.headerId) {
                            AdapterItem.Header(sectionOfAfter)
                        } else {
                            null
                        }
                    } else {
                        null
                    }
                }
            }
        }
        .cachedIn(viewModelScope)

    private var pendingInvalidation = false
    fun invalidateAfterDelay() {
        if (pendingInvalidation) {
            Log.d("NoteViewModel", "Already pending invalidation")
        } else {
            Log.d("NoteViewModel", "invalidateAfterDelay()")
            viewModelScope.launch {
                pendingInvalidation = true
                delay(250)
                Log.d("NoteViewModel", "invalidateAfterDelay() -> Invalidating now...")
                pagingSourceInvalidator?.invalidateNow("scroll")
                pendingInvalidation = false
            }
        }
    }

    private fun createData(amount: Int) {
        viewModelScope.launch(Dispatchers.IO) {
            var notesInDay = 0
            val notes = mutableListOf<Note>()
            val calendar = Calendar.getInstance()
            calendar.time = Date()

            for (i in 1..amount) {

                val noteText = i.toString()
                val date = calendar.timeInMillis
                notes.add(Note(note = noteText, date = date))
                notesInDay++

                if (notesInDay == 30) {
                    calendar.add(Calendar.DAY_OF_MONTH, -1)
                    notesInDay = 0
                }
            }

            repository.insertNotes(notes)
            prefs.setBooleanValue("isDataCreated", true)
        }

    }

    private fun getSection(note: Note): NoteHeader {
        val id = DateTimeFormatter.BASIC_ISO_DATE.withZone(ZoneId.systemDefault()).format(Instant.ofEpochMilli(note.date))
        val title = LocalDateTime.ofInstant(Instant.ofEpochMilli(note.date), ZoneId.systemDefault())
            .format(
                DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
                    .withLocale(Locale.getDefault())
            )

        return NoteHeader(id, title)
    }


    @Suppress("UNCHECKED_CAST")
    class Factory(private val app: Application) : ViewModelProvider.Factory {
        override fun <T : ViewModel> create(modelClass: Class<T>): T {
            return NoteViewModel(app) as T
        }
    }
}