package com.example.myapp.feature.home.list

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ImageSearch
import androidx.compose.material3.BottomAppBar
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.FloatingActionButtonDefaults.bottomAppBarFabElevation
import androidx.compose.material3.FloatingActionButtonElevation
import androidx.compose.material3.Icon
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.isTraversalGroup
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.tooling.preview.Preview
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.example.myapp.core.design_system.type.AppTheme

@Composable
fun HomeListScreen(viewModel: HomeListViewModel) {
    val uiState by viewModel.uiState.collectAsStateWithLifecycle()

    HomeListScreen(
        uiState = uiState,
    )
}

@Composable
private fun HomeListScreen(
    uiState: HomeListUiState,
) {
    val snackbarHostState = remember { SnackbarHostState() }

    @Composable
    fun fab(elevation: FloatingActionButtonElevation) {
        MainFab(
            elevation = elevation,
            onStartOperation = uiState.onStartOperationTapped
        )
    }

    Scaffold(
        snackbarHost = { SnackbarHost(snackbarHostState) },
        bottomBar = {
            MainBottomBar(
                fab = {
                    fab(elevation = bottomAppBarFabElevation())
                },
            )
        },
    ) { innerPadding ->
        Box(
            Modifier
                // consume insets as scaffold doesn't do it by default
                .consumeWindowInsets(innerPadding)
                .padding(innerPadding)
        ) {
            ItemsLoadedLayout(
                items = uiState.items,
            )
        }
    }

    MyControls(
        myState = uiState.myState,
        elapsedTime = uiState.elapsedTime,
        onPause = uiState.onPauseTapped,
        onResume = uiState.onResumeTapped,
        onStop = uiState.onStopTapped
    )
}

@Composable
private fun MainFab(
    elevation: FloatingActionButtonElevation,
    onStartOperation: () -> Unit,
) {
    Box {
        FloatingActionButton(
            elevation = elevation,
            onClick = {
                onStartOperation()
            },
            content = {
                Icon(
                    imageVector = Icons.Default.ImageSearch,
                    contentDescription = null
                )
            }
        )
    }
}

@Composable
private fun MainBottomBar(
    fab: @Composable () -> Unit,
) {
    BottomAppBar(
        actions = {},
        floatingActionButton = {
            fab()
        }
    )
}

@Composable
private fun ItemsLoadedLayout(
    items: List<HomeListItem>,
) {
    Box(
        Modifier
            .fillMaxSize()
            .semantics { isTraversalGroup = true }
    ) {
        Column {
            items.forEach {
                Text(it.name)
            }
        }
    }
}


@Preview
@Composable
private fun HomeScreenPreview() {
    AppTheme {
        HomeListScreen(
            uiState = HomeListUiState(),
        )
    }
}