
package com.example.myapp.feature.home.list

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import com.example.myapp.core.data.repository.MyRepository
import com.example.myapp.core.models.MyState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.sample
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import kotlin.random.Random
import kotlin.time.Duration

data class HomeListItem(val name: String)

data class HomeListUiState(
    val items: List<HomeListItem> = emptyList(),
    val myState: MyState = MyState.Stopped,
    val elapsedTime: Duration = Duration.ZERO,
    val currentValue: Float? = null,
    val onStartOperationTapped: () -> Unit = {},
    val onPauseTapped: () -> Unit = {},
    val onResumeTapped: () -> Unit = {},
    val onStopTapped: () -> Unit = {},
)

class HomeListViewModel(
    private val myRepository: MyRepository,
) : ViewModel() {
    private var updateElapsedTimeJob: Job? = null
    private val _items = MutableStateFlow(listOf(HomeListItem("One"), HomeListItem("Two"), HomeListItem("Three")))
    private val _currentTime = MutableStateFlow(Duration.ZERO)
    private val _values = MutableStateFlow(0f)

    @OptIn(FlowPreview::class)
    val uiState: StateFlow<HomeListUiState> = combine(
        _items,
        myRepository.state,
        _currentTime,
        _values.sample(33).distinctUntilChanged()
    ) { items, state, elapsedTime, currentValue ->
        HomeListUiState(
            items = items,
            myState = state,
            elapsedTime = elapsedTime,
            currentValue = currentValue,
            onStartOperationTapped = {
                viewModelScope.launch {
                    myRepository.start()
                }
            },
            onPauseTapped = {
                viewModelScope.launch {
                    myRepository.pause()
                }
            },
            onResumeTapped = {
                viewModelScope.launch {
                    myRepository.start()
                }
            },
            onStopTapped = {
                viewModelScope.launch {
                    myRepository.stop()
                }
            }
        )
    }.stateInViewModel(viewModelScope, HomeListUiState())

    init {
        myRepository.state
            .onEach { state ->
                when (state) {
                    MyState.Started -> {
                        updateElapsedTimeJob?.cancel()
                        updateElapsedTimeJob = viewModelScope.launch {
                            while (true) {
                                _values.update { Random.nextFloat() }
                                _currentTime.update {
                                    myRepository.getCurrentTime()
                                }
                                delay(250)
                            }
                        }
                    }

                    else -> {
                        updateElapsedTimeJob?.cancel()
                        updateElapsedTimeJob = null
                    }
                }
            }
            .launchIn(viewModelScope)
    }

    companion object {
        fun factory() = viewModelFactory {
            initializer {
                return@initializer HomeListViewModel(
                    myRepository = MyRepository(),
                )
            }
        }
    }
}

fun <T> Flow<T>.stateInViewModel(scope: CoroutineScope, initialValue: T): StateFlow<T> {
    return stateIn(
        scope = scope,
        started = SharingStarted.WhileSubscribed(stopTimeoutMillis = 5_000),
        initialValue = initialValue,
    )
}