package com.example.myapp.feature.home.list

import android.text.format.DateUtils
import android.util.Log
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FilledTonalButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.ModalBottomSheetDefaults
import androidx.compose.material3.SheetState
import androidx.compose.material3.SheetValue
import androidx.compose.material3.Text
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.unit.dp
import com.example.myapp.core.models.MyState
import kotlin.time.Duration

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MyControls(
    myState: MyState,
    elapsedTime: Duration,
    onPause: () -> Unit,
    onResume: () -> Unit,
    onStop: () -> Unit,
) {
    var showTheSheet by rememberSaveable { mutableStateOf(false) }
    val sheetState = rememberModalBottomSheetState(
        skipPartiallyExpanded = true,
        confirmValueChange = { sheetValue ->
            val ret = when (sheetValue) {
                SheetValue.Hidden -> false
                SheetValue.Expanded -> true
                SheetValue.PartiallyExpanded -> false
            }
            Log.d("TESTING", "confirmValueChange: sheetValue = $sheetValue, ret = $ret")
            ret
        }
    )

    // Handle state changes
    LaunchedEffect(myState) {
        Log.d("TESTING", "LaunchedEffect(myState), myState = $myState")
        when (myState) {
            MyState.Started -> showTheSheet = true
            MyState.Paused -> showTheSheet = true
            MyState.Stopped -> {
                sheetState.hide()
                // We have to remove it from the composition
                showTheSheet = false
            }
        }
    }

    // We have to remove it from the composition
    if (showTheSheet) {
        MyBottomSheet(
            sheetState = sheetState,
            isPaused = myState == MyState.Paused,
            elapsedTime = elapsedTime,
            onPauseTapped = onPause,
            onResumeTapped = onResume,
            onStopTapped = onStop
        )
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun MyBottomSheet(
    sheetState: SheetState,
    isPaused: Boolean,
    elapsedTime: Duration,
    onPauseTapped: () -> Unit,
    onResumeTapped: () -> Unit,
    onStopTapped: () -> Unit,
) {
    val inWholeSeconds = remember(elapsedTime) { elapsedTime.inWholeSeconds }

    ModalBottomSheet(
        sheetState = sheetState,
        dragHandle = null,
        properties = ModalBottomSheetDefaults.properties(shouldDismissOnBackPress = false),
        onDismissRequest = { /* Dismiss requests have to be handled via the stop button.  */ }
    ) {
        Column(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp),
            verticalArrangement = Arrangement.spacedBy(16.dp),
            horizontalAlignment = Alignment.CenterHorizontally,
        ) {

            Text(
                text = if (isPaused) {
                    "Paused"
                } else {
                    DateUtils.formatElapsedTime(inWholeSeconds)
                },
                style = MaterialTheme.typography.labelSmall
            )
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                // Not shown -- only for spacing purposes
                FilledTonalButton(
                    modifier = Modifier.alpha(0f),
                    onClick = { },
                    enabled = false
                ) {
                    Text("Stop")
                }
                Button(
                    onClick = {
                        if (isPaused) {
                            onResumeTapped()
                        } else {
                            onPauseTapped()
                        }
                    }
                ) {
                    Text(if (isPaused) "Resume" else "Stop")
                }
                FilledTonalButton(onClick = {
                    onStopTapped()
                }) {
                    Text("Stop")
                }
            }
        }
    }
}