package de.davidartmann.rosa.adapter;

import android.content.Context;
import android.support.v4.content.ContextCompat;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;
import java.util.List;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.adapter.viewholder.MainListArchivViewholder;
import de.davidartmann.rosa.db.async.FindAllInactivePersonsFromDb;
import de.davidartmann.rosa.model.Person;

/**
 * Adapter for the Archiv of inactive Persons.
 * Created by david on 06.03.16.
 */
public class MainListArchivAdapter extends RecyclerView.Adapter<MainListArchivViewholder>
        implements FindAllInactivePersonsFromDb.IFindAllInactivePersonsFromDb,
        MainListArchivViewholder.IMainListArchivViewholder {

    private int mLayout;
    private List<Person> mPersons;
    private Context mContext;
    private boolean mIsLongClicked;
    private int mNumberOfPersons;
    private IArchivFragmentLongClickMenu iArchivFragmentLongClickMenu;
    private List<Person> mPersonsToUnarchive;

    public MainListArchivAdapter(int layout, Context context,
                                 IArchivFragmentLongClickMenu iArchivFragmentLongClickMenu) {
        mIsLongClicked = false;
        mLayout = layout;
        mContext = context;
        mPersons = new ArrayList<>();
        mPersonsToUnarchive = new ArrayList<>();
        new FindAllInactivePersonsFromDb(context, this).execute();
        mNumberOfPersons = 0;
        this.iArchivFragmentLongClickMenu = iArchivFragmentLongClickMenu;
    }

    @Override
    public MainListArchivViewholder onCreateViewHolder(ViewGroup parent, int viewType) {
        View view = LayoutInflater.from(parent.getContext()).inflate(mLayout, parent, false);
        return new MainListArchivViewholder(view, mContext, this/*, this, this*/);
    }

    @Override
    public void onBindViewHolder(MainListArchivViewholder holder, int position) {
        holder.assignData(mPersons.get(position));
    }

    @Override
    public int getItemCount() {
        return mPersons.size();
    }

    @Override
    public void onFindAllInactive(List<Person> persons) {
        if (this.mPersons != null) {
            this.mPersons.clear();
            this.mPersons.addAll(persons);
        } else {
            this.mPersons = persons;
        }
        notifyDataSetChanged();
    }

    @Override
    public void onUpdated(int adapterPosition) {
        mPersons.remove(adapterPosition);
        notifyItemRemoved(adapterPosition);
    }

    @Override
    public void onLongClicked(int position, View view, Person person) {
        mIsLongClicked = true;
        view.setBackgroundColor(ContextCompat.getColor(mContext, R.color.colorLightGray));
        view.setElevation(1);
        mNumberOfPersons++;
        activatePerson(person);
        mPersonsToUnarchive.add(person);
        iArchivFragmentLongClickMenu.onLongClicked(true, mPersonsToUnarchive);
    }

    @Override
    public void onClick(int position, View view, Person person, int bgColor) {
        if (mIsLongClicked) {
            if (mNumberOfPersons != 0) {
                int colorBgGray = ContextCompat.getColor(mContext, R.color.colorLightGray);
                if (view.getElevation() == 1) {
                    view.setElevation(3);
                    setCardBackgroundByCategory(view, person);
                    mNumberOfPersons--;
                    deactivatePerson(person);
                    mPersonsToUnarchive.remove(person);
                    if (mNumberOfPersons == 0) {
                        iArchivFragmentLongClickMenu.onLongClicked(false, mPersonsToUnarchive);
                    }
                } else {
                    view.setBackgroundColor(colorBgGray);
                    view.setElevation(1);
                    mNumberOfPersons++;
                    activatePerson(person);
                    mPersonsToUnarchive.add(person);
                }
            } else {
                iArchivFragmentLongClickMenu.onLongClicked(false, mPersonsToUnarchive);
            }
        }
    }

    private void activatePerson(Person person) {
        person.setActive(true);
        person.setPosition(getItemCount()+1);
    }

    private void deactivatePerson(Person person) {
        person.setActive(false);
        person.setPosition(0);
    }

    private void setCardBackgroundByCategory(View view, Person person) {
        if (person.getCategory() == 0) {
            view.setBackgroundColor(ContextCompat.getColor(mContext, R.color.colorSure));
        } else if (person.getCategory() == 1) {
            view.setBackgroundColor(
                    ContextCompat.getColor(mContext, R.color.colorAlmostSure));
        } else if (person.getCategory() == 2) {
            view.setBackgroundColor(ContextCompat.getColor(mContext, R.color.colorUnsure));
        }
    }

    public interface IArchivFragmentLongClickMenu {
        void onLongClicked(boolean isActive, List<Person> personsToUnarchive);
    }
}
