package de.davidartmann.rosa.adapter.viewholder;

import android.content.Context;
import android.content.DialogInterface;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.CardView;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import com.squareup.picasso.Picasso;

import java.util.List;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.db.async.FindAllActivePersonsFromDb;
import de.davidartmann.rosa.db.async.UpdatePersonInDb;
import de.davidartmann.rosa.model.Person;

/**
 * Viewholder for the inactive Persons.
 * Created by david on 06.03.16.
 */
public class MainListArchivViewholder extends RecyclerView.ViewHolder
        implements View.OnClickListener, UpdatePersonInDb.IUpdatePersonInDb,
        FindAllActivePersonsFromDb.IFindAllActive, View.OnLongClickListener {

    private ImageView imageViewPicture;
    private TextView textViewName;
    private TextView textViewAddress;
    private Context mContext;
    private CardView cardView;
    private Person mPerson;
    private IMainListArchivViewholder iMainListArchivViewholder;
    private int mBgColor;

    public MainListArchivViewholder(View itemView, Context context,
                                    IMainListArchivViewholder iMainListArchivViewholder) {
        super(itemView);
        this.mContext = context;
        imageViewPicture = (ImageView) itemView
                .findViewById(R.id.listview_archiv_viewholder_picture);
        textViewName = (TextView) itemView.findViewById(R.id.listview_archiv_viewholder_name);
        textViewAddress = (TextView) itemView.findViewById(R.id.listview_archiv_viewholder_address);
        ImageView imageViewUnarchive = (ImageView) itemView
                .findViewById(R.id.listview_archiv_viewholder_unarchive);
        cardView = (CardView) itemView.findViewById(R.id.listview_archiv_viewholder_cardview);
        imageViewUnarchive.setOnClickListener(this);
        this.iMainListArchivViewholder = iMainListArchivViewholder;
        cardView.setOnLongClickListener(this);
        cardView.setOnClickListener(this);
    }

    public void assignData(Person person) {
        mPerson = person;
        textViewAddress.setText(person.getAddress());
        textViewName.setText(person.getName());
        if (person.getCategory() == 0) {
            mBgColor = ContextCompat.getColor(mContext, R.color.colorSure);
        } else if (person.getCategory() == 1) {
            mBgColor = ContextCompat.getColor(mContext, R.color.colorAlmostSure);
        } else if (person.getCategory() == 2) {
            mBgColor = ContextCompat.getColor(mContext, R.color.colorUnsure);
        }
        cardView.setCardBackgroundColor(mBgColor);
        Picasso.with(mContext).load(person.getPictureUrl()).placeholder(R.drawable.ic_face_24dp)
                .into(imageViewPicture);
    }

    @Override
    public void onClick(View v) {
        switch (v.getId()) {
            case R.id.listview_archiv_viewholder_unarchive:
                showRestoreDialog();
                break;
            case R.id.listview_archiv_viewholder_cardview:
                iMainListArchivViewholder.onClick(getAdapterPosition(), v, mPerson, mBgColor);
                break;
        }

    }

    private void showRestoreDialog() {
        AlertDialog.Builder builder = new AlertDialog.Builder(mContext);
        builder.setMessage(mContext.getString(R.string.dialog_archiv_message));
        builder.setPositiveButton(mContext.getString(R.string.Ja), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new FindAllActivePersonsFromDb(mContext, MainListArchivViewholder.this).execute();
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(mContext.getString(R.string.Nein), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    @Override
    public void onUpdate(Person person) {
        iMainListArchivViewholder.onUpdated(getAdapterPosition());
    }

    @Override
    public boolean onLongClick(View v) {
        iMainListArchivViewholder.onLongClicked(getAdapterPosition(), v, mPerson);
        return true;
    }

    @Override
    public void onFindAllActive(List<Person> persons) {
        mPerson.setActive(true);
        mPerson.setPosition(persons.size()+1);
        new UpdatePersonInDb(mContext, MainListArchivViewholder.this).execute(mPerson);
    }

    public interface IMainListArchivViewholder {
        void onUpdated(int adapterPosition);
        void onLongClicked(int position, View view, Person person);
        void onClick(int position, View view, Person person, int bgColor);
    }
}
