package de.davidartmann.rosa.fragment;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.provider.MediaStore;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.Snackbar;
import android.support.v4.app.ActivityCompat;
import android.support.v4.app.Fragment;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AlertDialog;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.EditText;
import android.widget.ImageView;

import com.squareup.picasso.Picasso;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.activity.MainListActivity;
import de.davidartmann.rosa.db.async.CreateOrUpdatePerson;
import de.davidartmann.rosa.db.async.UpdatePersonInDb;
import de.davidartmann.rosa.model.Person;

/**
 * Fragment for the edit-view of a detailed persons view.
 * Created by david on 03.03.16.
 */
public class DetailEditFragment extends Fragment implements View.OnClickListener,
        UpdatePersonInDb.IUpdatePersonInDb {

    private static final String TAG = DetailEditFragment.class.getSimpleName();
    private static final int REQUEST_IMAGE_CAPTURE = 1;
    private static final int REQUEST_IMAGE_PICK = 2;
    private static final int REQUEST_PERMISSION_READ_EXTERNAL_STORAGE = 3;

    private EditText editTextName;
    private EditText editTextPhone;
    private EditText editTextEmail;
    private EditText editTextAddress;
    private EditText editTextPrice;
    private EditText editTextMisc;
    private ImageView mImageViewPicture;
    private Person mPerson;
    private boolean isUpdate;
    private int mNumberOfItemsInMainList;
    private String mCurrentPicturePath;

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_detail_edit, container, false);
        setHasOptionsMenu(true);
        isUpdate = false;
        editTextName = (EditText) view.findViewById(R.id.fragment_detail_edit_name);
        editTextPhone = (EditText) view.findViewById(R.id.fragment_detail_edit_phone);
        editTextEmail = (EditText) view.findViewById(R.id.fragment_detail_edit_email);
        editTextAddress = (EditText) view.findViewById(R.id.fragment_detail_edit_address);
        editTextPrice = (EditText) view.findViewById(R.id.fragment_detail_edit_price);
        editTextMisc = (EditText) view.findViewById(R.id.fragment_detail_edit_misc);
        mImageViewPicture = (ImageView) view.findViewById(R.id.fragment_detail_edit_picture);
        Bundle bundle = getArguments();
        if (bundle != null) {
            mPerson = (Person) bundle.getSerializable(DetailFragment.PERSON);
            if (mPerson != null) {
                assignData(mPerson);
                isUpdate = true;
            }
            mNumberOfItemsInMainList = bundle.getInt(DetailFragment.POSITION);

        }
        resetActionBar(true);
        mImageViewPicture.setOnClickListener(this);
        return view;
    }

    private void assignData(Person person) {
        editTextName.setText(person.getName());
        editTextPhone.setText(person.getPhone());
        editTextEmail.setText(person.getEmail());
        editTextAddress.setText(person.getAddress());
        editTextPrice.setText(person.getPrice());
        editTextMisc.setText(person.getMisc());
        Picasso.with(getContext()).load(person.getPictureUrl()).into(mImageViewPicture);
    }

    public static DetailEditFragment newInstanceForUpdate(Person person) {
        DetailEditFragment detailEditFragment = new DetailEditFragment();
        Bundle bundle = new Bundle();
        bundle.putSerializable(DetailFragment.PERSON, person);
        detailEditFragment.setArguments(bundle);
        return detailEditFragment;
    }

    public static DetailEditFragment newInstanceForCreate(int position) {
        DetailEditFragment detailEditFragment = new DetailEditFragment();
        Bundle bundle = new Bundle();
        bundle.putInt(DetailFragment.POSITION, position);
        detailEditFragment.setArguments(bundle);
        return detailEditFragment;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        menu.clear();
        inflater.inflate(R.menu.menu_detail_edit, menu);
        MenuItem menuItem = menu.findItem(R.id.menu_detail_edit_save);
        menuItem.setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);
        menuItem.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem item) {
                showCategoryPicker();
                return true;
            }
        });
    }

    private Person getPersonFromFieldValues(int which) {
        if (mPerson == null) {
            mPerson = new Person();
        }
        if (mNumberOfItemsInMainList != 0) {
            mPerson.setPosition(mNumberOfItemsInMainList+1);
        }
        mPerson.setPrice(editTextPrice.getText().toString().trim());
        mPerson.setActive(true);
        mPerson.setMisc(editTextMisc.getText().toString().trim());
        mPerson.setAddress(editTextAddress.getText().toString().trim());
        mPerson.setCategory(which);
        mPerson.setEmail(editTextEmail.getText().toString().trim());
        mPerson.setName(editTextName.getText().toString().trim());
        if (mCurrentPicturePath != null) {
            mPerson.setPictureUrl(mCurrentPicturePath);
        }
        mPerson.setPhone(editTextPhone.getText().toString().trim());
        return mPerson;
    }

    private void showCategoryPicker() {
        Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle(getString(R.string.Wähle_Kategorie));
        final ArrayAdapter<String> arrayAdapter = new ArrayAdapter<>(context,
                android.R.layout.simple_list_item_single_choice);
        arrayAdapter.addAll(getResources().getStringArray(R.array.categories));
        int category = isUpdate ? mPerson.getCategory() : 0;
        builder.setSingleChoiceItems(arrayAdapter, category, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new CreateOrUpdatePerson(getContext(), isUpdate)
                        .execute(getPersonFromFieldValues(which));
                dialog.dismiss();
                getActivity().getSupportFragmentManager().beginTransaction()
                        .replace(R.id.activity_main_framelayout, new MainListFragment())
                        .commit();
                resetActionBar(false);
            }
        });
        builder.setNegativeButton("Abbrechen", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    private void resetActionBar(boolean b) {
        ActionBar actionBar = ((MainListActivity)getActivity()).getSupportActionBar();
        if (actionBar != null) {
            actionBar.setDisplayHomeAsUpEnabled(b);
            actionBar.setHomeButtonEnabled(b);
            if (isUpdate) {
                actionBar.setTitle(R.string.Person_bearbeiten);
            } else {
                actionBar.setTitle(R.string.Neue_Person);
            }
        }
    }

    private Intent getPickImageIntent() {
        Intent intent = new Intent();
        intent.setType("image/*");
        intent.setAction(Intent.ACTION_GET_CONTENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        return Intent.createChooser(intent, "Select picture");
    }

    @Override
    public void onClick(View v) {
        if (deviceHasCamera()) {
            showPictureDialog();
        } else {
            showSnackBar("Leider verfügt ihr Gerät über keine Kamera", "device has no camera");
        }
    }

    private void showPictureDialog() {
        final Context context = getContext();
        String[] text = new String[] {"Mit Kamera aufnehmen", "Aus Galerie auswählen",
                "Bild löschen"};
        CharSequence[] items;
        if (mPerson.getPictureUrl() == null) {
            items = new CharSequence[2];
        } else {
            items = new CharSequence[3];
        }
        System.arraycopy(text, 0, items, 0, items.length);
        AlertDialog.Builder builder	= new AlertDialog.Builder(context);
        builder.setTitle("Was möchten Sie tun?");
        builder.setSingleChoiceItems(items, 0, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                if (which == 0) {
                    Intent takePictureIntent = new Intent(MediaStore.ACTION_IMAGE_CAPTURE);
                    if (takePictureIntent
                            .resolveActivity(getActivity().getPackageManager()) != null) {
                        File photoFile = createImageFile();
                        if (photoFile != null) {
                            takePictureIntent.putExtra(MediaStore.EXTRA_OUTPUT,
                                    Uri.fromFile(photoFile));
                            startActivityForResult(takePictureIntent, REQUEST_IMAGE_CAPTURE);
                            dialog.dismiss();
                        }
                    }
                } else if (which == 1) {
                    requestReadExternalStorage();
                    dialog.dismiss();
                } else {
                    deleteOldImgFile();
                    mPerson.setPictureUrl(null);
                    Picasso.with(context).load(R.drawable.ic_face_24dp).into(mImageViewPicture);
                    new UpdatePersonInDb(context, DetailEditFragment.this).execute(mPerson);
                    dialog.dismiss();
                }
            }
        });
        builder.create().show();
    }

    private void requestReadExternalStorage() {
        String permission = Manifest.permission.READ_EXTERNAL_STORAGE;
        Activity activity = getActivity();
        if (Build.VERSION.SDK_INT == Build.VERSION_CODES.M) {
            if (ContextCompat.checkSelfPermission(
                    activity, permission) != PackageManager.PERMISSION_GRANTED) {
                // Explain to the user why we need to read the contacts (NOT NEEDED!)
                //if (ActivityCompat.shouldShowRequestPermissionRationale(activity,
                //        Manifest.permission.READ_EXTERNAL_STORAGE)) {}
                ActivityCompat.requestPermissions(getActivity(), new String[]{permission},
                        REQUEST_PERMISSION_READ_EXTERNAL_STORAGE);
            } else {
                startActivityForResult(getPickImageIntent(), REQUEST_IMAGE_PICK);
            }
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions,
                                           @NonNull int[] grantResults) {
        switch (requestCode) {
            case REQUEST_PERMISSION_READ_EXTERNAL_STORAGE:
                if (grantResults.length > 0
                        && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                    startActivityForResult(getPickImageIntent(), REQUEST_IMAGE_PICK);
                } else {
                    Log.d(TAG, "permission READ_EXTERNAL_STORAGE not granted");
                    // what to do if the permissions were not granted...
                    //mImageViewPicture.setOnLongClickListener(null);
                }

        }
    }

    private File createImageFile() {
        String timeStamp = new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.GERMANY)
                .format(new Date());
        String imageFileName = "JPEG_" + timeStamp + "_";
        Context context = getContext();
        // the media scanner cannot access the files because they are private to our app
        // this is the right directory as base for relative image paths for the future
        File storageDir = context.getExternalFilesDir(Environment.DIRECTORY_PICTURES);
        File image = null;
        try {
            image = File.createTempFile(
                    imageFileName,  /* prefix */
                    ".jpg",         /* suffix */
                    storageDir      /* directory */
            );
            // Save a file: path for use with ACTION_VIEW intents
            //TODO: maybe relative paths are better, this needs to be tested
            //Log.d(TAG, "path="+image.getPath());
            //Log.d(TAG, "absolutePath="+image.getAbsolutePath());
            //Log.d(TAG, image.getPath().substring(image.getPath().indexOf("data/")));
            //Log.d(TAG, Environment.getDataDirectory().getAbsolutePath());
            mCurrentPicturePath = "file:" + image.getAbsolutePath();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return image;
    }

    private void showSnackBar(String snackBarMsg, String fallBackMsg) {
        View view = getView();
        if (view != null) {
            Snackbar.make(view, snackBarMsg, Snackbar.LENGTH_LONG).show();
        } else {
            Log.w(TAG, fallBackMsg);
        }
    }

    private boolean deviceHasCamera() {
        return getActivity().getPackageManager().hasSystemFeature(PackageManager.FEATURE_CAMERA);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            Context context = getContext();
            if (requestCode == REQUEST_IMAGE_CAPTURE) {
                /**
                 * this would give us a thumbnail
                 *
                Bundle extras = data.getExtras();
                Bitmap imageBitmap = (Bitmap) extras.get("data");
                mImageViewPicture.setImageBitmap(imageBitmap);
                */
                deleteOldImgFile();// so the old file will not be unused, but this needs be checked by asking the user
                mPerson.setPictureUrl(mCurrentPicturePath);
                Picasso.with(context).load(mCurrentPicturePath).into(mImageViewPicture);
                new UpdatePersonInDb(context, this).execute(mPerson);
            } else if (requestCode == REQUEST_IMAGE_PICK) {
                Uri selectedImageUri = data.getData();
                mPerson.setPictureUrl(selectedImageUri.toString());
                Picasso.with(context).load(selectedImageUri).into(mImageViewPicture);
                new UpdatePersonInDb(context, this).execute(mPerson);
            }
        }
    }

    private void deleteOldImgFile() {
        if (mPerson.getPictureUrl() != null) {
            int index = mPerson.getPictureUrl().indexOf("/storage");
            if (index > -1) {
                File file = new File(mPerson.getPictureUrl().substring(index));
                boolean isDeleted = file.delete();
                if (!isDeleted) {
                    Log.w(TAG, "file could not be deleted");
                }
            } else {
                Log.i(TAG, "no self made picture, so nothing to delete");
            }
        }
    }

    @Override
    public void onUpdate(Person person) {
        mPerson = person;
    }
}
