package de.davidartmann.rosa.fragment;

import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.design.widget.FloatingActionButton;
import android.support.v4.app.Fragment;
import android.support.v7.app.AlertDialog;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.ImageView;
import android.widget.TextView;

import com.squareup.picasso.Picasso;

import java.util.List;

import de.davidartmann.rosa.activity.MainListActivity;
import de.davidartmann.rosa.R;
import de.davidartmann.rosa.db.async.DecrementPositionOfAllPersonsFromDb;
import de.davidartmann.rosa.db.async.FindAllActivePersonsFromDb;
import de.davidartmann.rosa.db.async.SoftDeletePersonInDb;
import de.davidartmann.rosa.db.async.UpdatePersonInDb;
import de.davidartmann.rosa.model.Person;

/**
 * Fragment for the detail view of a person.
 * Created by david on 03.03.16.
 */
public class DetailFragment extends Fragment implements View.OnClickListener,
        UpdatePersonInDb.IUpdatePersonInDb, MenuItem.OnMenuItemClickListener,
        FindAllActivePersonsFromDb.IFindAllActive,
        DecrementPositionOfAllPersonsFromDb.IDecrementPosition,
        SoftDeletePersonInDb.ISoftDeletePerson {

    public static final String PERSON = "person";
    public static final String EDIT = "edit";
    public static final String POSITION = "position";

    private TextView textViewName;
    private TextView textViewPhone;
    private TextView textViewEmail;
    private TextView textViewAddress;
    private TextView textViewPrice;
    private TextView textViewMisc;
    private ImageView imageViewPicture;
    private Person mPerson;
    private int mDecrementStartPos;

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_detail, container, false);
        textViewName = (TextView) view.findViewById(R.id.fragment_detail_name);
        textViewPhone = (TextView) view.findViewById(R.id.fragment_detail_phone);
        textViewEmail = (TextView) view.findViewById(R.id.fragment_detail_email);
        textViewAddress = (TextView) view.findViewById(R.id.fragment_detail_address);
        textViewPrice = (TextView) view.findViewById(R.id.fragment_detail_price);
        textViewMisc = (TextView) view.findViewById(R.id.fragment_detail_misc);
        imageViewPicture = (ImageView) view.findViewById(R.id.fragment_detail_picture);
        FloatingActionButton fabCategory = (FloatingActionButton) view.findViewById(R.id.fragment_detail_fab);
        fabCategory.setOnClickListener(this);
        mPerson = (Person) getArguments().getSerializable(PERSON);
        setHasOptionsMenu(true);
        assignData(mPerson);
        mDecrementStartPos = 0;
        return view;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        inflater.inflate(R.menu.menu_detail, menu);
        MenuItem deleteMenuItem = menu.findItem(R.id.menu_detail_delete);
        MenuItem editMenuItem = menu.findItem(R.id.menu_detail_edit);
        deleteMenuItem.setOnMenuItemClickListener(this);
        editMenuItem.setOnMenuItemClickListener(this);
    }

    public static DetailFragment newInstance(Person person) {
        DetailFragment detailFragment = new DetailFragment();
        Bundle bundle = new Bundle();
        bundle.putSerializable(PERSON, person);
        detailFragment.setArguments(bundle);
        return detailFragment;
    }

    private void assignData(Person person) {
        textViewName.setText(person.getName());
        textViewPhone.setText(person.getPhone());
        textViewEmail.setText(person.getEmail());
        textViewAddress.setText(person.getAddress());
        textViewPrice.setText(person.getPrice());
        textViewMisc.setText(person.getMisc());
        Picasso.with(getContext()).load(person.getPictureUrl()).into(imageViewPicture);
    }

    @Override
    public void onClick(View v) {
        switch (v.getId()) {
            case R.id.fragment_detail_fab:
                showCategoryPicker();
        }
    }

    private void showCategoryPicker() {
        Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle(getString(R.string.Wähle_Kategorie));
        final ArrayAdapter<String> arrayAdapter = new ArrayAdapter<>(context,
                android.R.layout.simple_list_item_single_choice);
        arrayAdapter.addAll(getResources().getStringArray(R.array.categories));
        builder.setSingleChoiceItems(arrayAdapter, mPerson.getCategory(),
                new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                // when category is changed, we want to store this new info
                new UpdatePersonInDb(getContext(), DetailFragment.this)
                        .execute(setNewCategoryOfPerson(which));
                dialog.dismiss();
            }
        });
        builder.setNegativeButton("Abbrechen", new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    private Person setNewCategoryOfPerson(int which) {
        mPerson.setCategory(which);
        return mPerson;
    }

    @Override
    public void onUpdate(Person person) {
        mPerson = person;
        assignData(person);
    }

    @Override
    public boolean onMenuItemClick(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.menu_detail_delete:
                showDeletionDialog();
                return true;
            case R.id.menu_detail_edit:
                showDetailEditFragment();
                return true;
        }
        return false;
    }

    private void showDetailEditFragment() {
        Context context = getContext();
        Intent intent = new Intent(context, MainListActivity.class);
        intent.putExtra(EDIT, true);
        intent.putExtra(PERSON, mPerson);
        //TODO: check this out, what is better destroy this instance or not -> also in the showDeletionDialog!
        //getActivity().finish();
        context.startActivity(intent);
    }

    private void showDeletionDialog() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setMessage(context.getString(R.string.dialog_delete_message));
        builder.setPositiveButton(context.getString(R.string.Ja), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                mDecrementStartPos = mPerson.getPosition();
                new SoftDeletePersonInDb(context, DetailFragment.this).execute(mPerson);
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(context.getString(R.string.Nein), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    @Override
    public void onDecremented(List<Person> persons) {
        Context context = getContext();
        Intent intent = new Intent(context, MainListActivity.class);
        //getActivity().finish();
        context.startActivity(intent);
    }

    @Override
    public void onSoftDeleted(Person person) {
        new FindAllActivePersonsFromDb(getContext(), this).execute();
    }

    @Override
    public void onFindAllActive(List<Person> persons) {
        new DecrementPositionOfAllPersonsFromDb(getContext(), this, persons, mDecrementStartPos).execute();
    }
}
