package com.example.animatedcontentbug.tab

import androidx.activity.compose.BackHandler
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.ProvidableCompositionLocal
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.staticCompositionLocalOf
import cafe.adriel.voyager.core.annotation.InternalVoyagerApi
import cafe.adriel.voyager.core.lifecycle.ScreenLifecycleStore
import cafe.adriel.voyager.navigator.Navigator
import cafe.adriel.voyager.navigator.NavigatorDisposeBehavior
import cafe.adriel.voyager.navigator.compositionUniqueId
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

typealias TabNavigatorContent = @Composable (tabNavigator: TabNavigator) -> Unit

val LocalTabNavigator: ProvidableCompositionLocal<TabNavigator> =
    staticCompositionLocalOf { error("TabNavigator not initialized") }

@OptIn(InternalVoyagerApi::class)
@Composable
fun TabNavigator(
    tab: Tab,
    backToFirst: Boolean = false,
    key: String = compositionUniqueId(),
    content: TabNavigatorContent = { CurrentTab() },
) {
    Navigator(tab, disposeBehavior = NavigatorDisposeBehavior(disposeNestedNavigators = false, disposeSteps = false), onBackPressed = null, key = key) { navigator ->
        val scope = rememberCoroutineScope()
        val tabNavigator = remember(navigator) {
            TabNavigator(navigator, scope)
        }
        DisposableEffect(navigator) {
            onDispose {
                tabNavigator.onDispose()
            }
        }
        CompositionLocalProvider(LocalTabNavigator provides tabNavigator) {
            if (backToFirst && tabNavigator.current != tab && runCatching { tabNavigator.current.isRoot.value }.getOrDefault(true)) {
                BackHandler(
                    onBack = {
                        tabNavigator.current = tab
                    },
                )
            }
            content(tabNavigator)
        }
    }
}

private val tabReselectHandlers = mutableListOf<suspend () -> Boolean>()

class TabNavigator internal constructor(
    private val navigator: Navigator,
    private val coroutineScope: CoroutineScope,
) {

    private val tabHistory = mutableSetOf(navigator.lastItem as Tab)

    fun onDispose() {
        tabHistory.forEach {
            it.onDispose()
            ScreenLifecycleStore.remove(it)
        }
    }

    var current: Tab
        get() = navigator.lastItem as Tab
        set(tab) {
            if (navigator.lastItem == tab) {
                coroutineScope.launch {
                    tabReselectHandlers.reversed().firstOrNull { it.invoke() }
                }
            } else {
                coroutineScope.launch(Dispatchers.Main.immediate) {
                    tabHistory.add(tab)
                    navigator.replaceAll(tab)
                }
            }
        }

    @Composable
    fun SaveableState(
        key: String,
        tab: Tab = current,
        content: @Composable () -> Unit,
    ) {
        navigator.saveableState(key, tab, content = content)
    }
}

@Composable
fun TabReselectHandler(handler: suspend () -> Boolean) {
    DisposableEffect(Unit) {
        tabReselectHandlers.add(handler)
        onDispose { tabReselectHandlers.remove(handler) }
    }
}
