# Overview

This project demonstrates a bug when building an APK with a large third-party AAR.
The error looks something like this:
```text
* What went wrong:
Execution failed for task ':app:collectDebugDependencies'.
> File /home/iosephknecht/AndroidStudioProjects/agpfittocontentbug/localMaven/io/github/iosephknecht/feature-1/1.0.0/feature-1-1.0.0.aar is too big (2326853975 bytes) to fit in memory.
```

# Scenario for reproducing the bug on real projects

1. Your project has a third-party runtime dependency on aar, which weighs more than 2GB
2. Your application has a build type with debuggable flag disabled (relevant for all release build types)

# Quick start

## Publish large third-party library

```shell
./gradlew :feature-1:debugMavenPublishFake
```

After executing publish command, a folder `localMaven` with the published aar will appear in the project folder.

## Build application

```shell
./gradlew :app:assembleDebug
```

After executing the build command, it is expected that it will fail and the error from the description will be output.

# Workaround fix

## Note

You can check the fix given below in the project, for this in the project 
file gradle.properties you need to set the flag:
```properties
# gradle.properties

# change this flag to true
android.toFitInMemoryFix=true
```

## Fix
```kt
// in your app build gradle
afterEvaluate {
    tasks.withType(PerModuleReportDependenciesTask::class.java) {
        val originalArtifactCollection = runtimeClasspathArtifacts.get()
        runtimeClasspathArtifacts.set(
            LargeFileFilteringArtifactCollection(originalArtifactCollection)
        )
    }
}

class LargeFileFilteringArtifactCollection(
    private val artifactCollection: ArtifactCollection
) : ArtifactCollection {

    override fun iterator(): MutableIterator<ResolvedArtifactResult> {
        val newIterator = artifactCollection
            .asSequence()
            .filter { result -> result.file.length() < Int.MAX_VALUE }
            .iterator()

        return object : MutableIterator<ResolvedArtifactResult> {

            override fun hasNext(): Boolean {
                return newIterator.hasNext()
            }

            override fun next(): ResolvedArtifactResult {
                return newIterator.next()
            }

            override fun remove() {
                error("unsupported")
            }
        }
    }

    override fun getArtifactFiles(): FileCollection {
        return artifactCollection
            .artifactFiles
            .filter { it.length() < Int.MAX_VALUE }
    }

    override fun getArtifacts(): MutableSet<ResolvedArtifactResult> {
        return artifactCollection
            .artifacts
            .filterTo(LinkedHashSet()) { result -> result.file.length() < Int.MAX_VALUE }
    }

    override fun getResolvedArtifacts(): Provider<MutableSet<ResolvedArtifactResult>> {
        return artifactCollection
            .resolvedArtifacts
            .map {
                it.filterTo(LinkedHashSet()) { result -> result.file.length() < Int.MAX_VALUE }
            }
    }

    override fun getFailures(): MutableCollection<Throwable> {
        return artifactCollection.failures
    }
}
```