import com.android.build.api.artifact.SingleArtifact
import com.android.build.gradle.tasks.BundleAar
import java.util.UUID

plugins {
    alias(libs.plugins.android.library)
    alias(libs.plugins.kotlin.android)
}

android {
    namespace = "io.github.iosephknecht.feature_1"
    compileSdk = 35

    defaultConfig {
        minSdk = 24

        testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"
        consumerProguardFiles("consumer-rules.pro")
    }

    buildTypes {
        release {
            isMinifyEnabled = false
            proguardFiles(
                getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro"
            )
        }
    }
    compileOptions {
        sourceCompatibility = JavaVersion.VERSION_11
        targetCompatibility = JavaVersion.VERSION_11
    }
    kotlinOptions {
        jvmTarget = "11"
    }
}

tasks.named<Delete>("clean") {
    delete("src/main/jniLibs")
    delete(rootProject.layout.projectDirectory.dir("localMaven"))
}

afterEvaluate {
    tasks.withType(BundleAar::class.java) {
        // hack so that aar does not compress
        entryCompression = ZipEntryCompression.STORED
    }
}

val generateFakeSoLibrariesTask =
    tasks.register<GenerateFakeSoLibraries>("generateFakeSoLibraries") {
        output.set(layout.projectDirectory.dir("src/main/jniLibs"))
    }

tasks.named("preBuild") {
    dependsOn(generateFakeSoLibrariesTask)
}

androidComponents {
    onVariants(selector = selector().withBuildType("debug")) { variant ->
        tasks.register<MavenPublishFake>("${variant.name}MavenPublishFake") {
            val aarProvider = variant.artifacts.get(SingleArtifact.AAR)
            aar.set(aarProvider)
            output.set(rootProject.layout.projectDirectory.dir("localMaven"))
        }
    }
}

abstract class GenerateFakeSoLibraries : DefaultTask() {

    @get:OutputDirectory
    abstract val output: DirectoryProperty

    @TaskAction
    fun taskAction() {
        val outputDir = output.get().asFile
        val arm64V8aDir = File(outputDir, "arm64-v8a")
        val armeabiV7a = File(outputDir, "armeabi-v7a")
        val x86Dir = File(outputDir, "x86")

        sequenceOf(arm64V8aDir, armeabiV7a, x86Dir)
            .filter { it.exists() }
            .forEach { it.deleteRecursively() }

        val fakeSoFile = File(outputDir, UUID.randomUUID().toString())

        fakeSoFile
            .bufferedWriter()
            .use { out ->
                for (i in 0 until 1_500_000 /* approximately 700 MB */) {
                    out.write(LOREM_IPSUM)
                }
            }

        try {
            sequenceOf(arm64V8aDir, armeabiV7a, x86Dir)
                .forEach {
                    it.mkdirs()
                    fakeSoFile.copyTo(File(it, "fake.so"), overwrite = true)
                }
        } finally {
            fakeSoFile.delete()
        }
    }

    private companion object {

        const val LOREM_IPSUM = """
            Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt
            ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco
            laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit 
            in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat 
            cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.
        """
    }
}

abstract class MavenPublishFake : DefaultTask() {

    @get:InputFile
    abstract val aar: RegularFileProperty

    @get:OutputDirectory
    abstract val output: DirectoryProperty

    @TaskAction
    fun taskAction() {
        val aarFile = aar.get().asFile
        val outputFile = output.get().asFile

        val libraryPath = File(outputFile, "io/github/iosephknecht/feature-1/1.0.0")
        libraryPath.mkdirs()

        val pomFile = File(libraryPath, "feature-1-1.0.0.pom")

        pomFile.writeText(
            """
            <?xml version="1.0" encoding="UTF-8"?>
            <project xsi:schemaLocation="http://maven.apache.org/POM/4.0.0 https://maven.apache.org/xsd/maven-4.0.0.xsd" xmlns="http://maven.apache.org/POM/4.0.0"
                xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
              <modelVersion>4.0.0</modelVersion>
              <groupId>io.github.iosephknecht</groupId>
              <artifactId>feature-1</artifactId>
              <version>1.0.0</version>
              <packaging>aar</packaging>
            </project>
            """.trimIndent()
        )

        aarFile.copyTo(File(libraryPath, "feature-1-1.0.0.aar"), overwrite = true)
    }
}