package com.example.myapplication

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.AccountCircle
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Build
import androidx.compose.material.icons.filled.Check
import androidx.compose.material.icons.filled.CheckCircle
import androidx.compose.material.icons.filled.DateRange
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Email
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import com.example.myapplication.ui.theme.MyApplicationTheme

data class ListItem(
    var title: String,
    val description: String,
    val icon: @Composable () -> Unit
)

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            MyApplicationTheme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    PopularListView(modifier = Modifier.padding(innerPadding))
                }
            }
        }
    }
}

@Composable
fun PopularListView(modifier: Modifier = Modifier) {
    val items = remember {
        mutableStateListOf(
            ListItem("Account", "Manage your account", { Icon(Icons.Filled.AccountCircle, contentDescription = "Account") }),
            ListItem("Tools", "Development tools", { Icon(Icons.Filled.Build, contentDescription = "Tools") }),
            ListItem("Completed", "Tasks completed", { Icon(Icons.Filled.CheckCircle, contentDescription = "Completed") }),
            ListItem("Date", "Calendar events", { Icon(Icons.Filled.DateRange, contentDescription = "Date") }),
            ListItem("Email", "Check your emails", { Icon(Icons.Filled.Email, contentDescription = "Email") })
        )
    }

    LazyColumn(
        modifier = modifier
            .fillMaxSize()
            .padding(16.dp),
        verticalArrangement = Arrangement.spacedBy(16.dp)
    ) {
        items(items) { item ->
            ListItemCard(item = item)
        }
        // ---- Remove ----
        item {
            IconButton(onClick = {
                items.add(ListItem("New Item", "Description", { Icon(Icons.Filled.Add, contentDescription = "New Item") }))
            }) {
                Icon(Icons.Filled.Add, contentDescription = "Add Item")
            }
        }
        // ---- Remove ----
    }
}

@Composable
fun ListItemCard(item: ListItem) {
    var isEditing by remember { mutableStateOf(false) }
    var editedTitle by remember { mutableStateOf(item.title) }

    Card(
        modifier = Modifier
            .fillMaxWidth()
            .clip(RoundedCornerShape(8.dp)),
        elevation = CardDefaults.cardElevation(defaultElevation = 4.dp)
    ) {
        Box(modifier = Modifier.fillMaxWidth()) {
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(16.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Box(
                    modifier = Modifier
                        .size(80.dp)
                        .clip(RoundedCornerShape(8.dp)),
                    contentAlignment = Alignment.Center
                ) {
                    item.icon()
                }
                Spacer(modifier = Modifier.width(16.dp))
                Column(modifier = Modifier.weight(1f)) {
                    if (isEditing) {
                        TextField(
                            value = editedTitle,
                            onValueChange = { editedTitle = it },
                            singleLine = true,
                            label = { Text("Edit Title") },
                            modifier = Modifier.fillMaxWidth()
                        )
                    } else {
                        Text(
                            text = item.title,
                            style = MaterialTheme.typography.headlineSmall,
                            fontWeight = FontWeight.Bold
                        )
                    }
                    Text(
                        text = item.description,
                        style = MaterialTheme.typography.bodyMedium,
                        color = Color.Gray
                    )
                }
            }
            IconButton(
                onClick = {
                    if (isEditing) {
                        item.title = editedTitle
                    }
                    isEditing = !isEditing
                },
                modifier = Modifier.align(Alignment.TopEnd)
            ) {
                Icon(
                    imageVector = if (isEditing) Icons.Filled.Check else Icons.Filled.Edit,
                    contentDescription = if (isEditing) "Finish Editing" else "Edit"
                )
            }
        }
    }
}

@Preview(showBackground = true)
@Composable
fun PopularListViewPreview() {
    MyApplicationTheme {
        PopularListView()
    }
}