package com.example.windowinsetstest.ui.navigation

import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.navigation.NavGraphBuilder
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import com.example.windowinsetstest.ui.navigation.NavigationEvent.Companion.onEvent

abstract class ComposeNavigation<T: NavigationComponent<*>, NavEvent> {
    abstract val startDestination: T
    abstract val composables: List<NavigationComposable<*, out T, NavEvent>>

    fun NavGraphBuilder.setComponents(
        onNavigationEvent: (NavigationEvent) -> Unit,
        onEvent: (NavEvent) -> Unit
    ){
        composables.forEach { composable ->
            with(composable) {
                when (this) {
                    is DestinationComposable -> composable(
                        onNavigationEvent = onNavigationEvent,
                        onGraphEvent = onEvent,
                    )

                    is GraphComposable -> setGraph(
                        onNavigationEvent = onNavigationEvent,
                        onEvent = onEvent
                    )

                    is DestinationObjectComposable -> composable(
                        onNavigationEvent = onNavigationEvent,
                        onGraphEvent = onEvent,
                    )
                }
            }
        }
    }

    /**
     * Compose the [ComposeNavigation] along with a [NavHost]
     *
     * If a [NavHost] already exists, [setComponents] can be used for more tailored usage.
     */
    @Composable
    fun NavigationContent(
        modifier: Modifier = Modifier,
        controller: NavHostController,
        onEvent: (NavEvent) -> Unit = {
            error(
                buildString {
                    append("Unhandled Navigation: ")
                    append(this@ComposeNavigation)
                    appendLine()
                    append("Event: ")
                    append(it)
                }
            )
        }
    ) {
        NavHost(
            navController = controller,
            modifier = modifier,
            startDestination = startDestination
        ){
            setComponents(
                onNavigationEvent = {
                    controller.onEvent(it)
                },
                onEvent = onEvent
            )
        }
    }
}