package com.example.windowinsetstest.ui.navigation

import android.annotation.SuppressLint
import android.os.Bundle
import androidx.compose.animation.AnimatedContentTransitionScope
import androidx.compose.animation.EnterTransition
import androidx.compose.animation.ExitTransition
import androidx.navigation.NavBackStackEntry
import androidx.navigation.NavDeepLink
import androidx.navigation.NavGraphBuilder
import androidx.navigation.NavType
import androidx.navigation.compose.ComposeNavigator
import androidx.navigation.compose.ComposeNavigatorDestinationBuilder
import androidx.navigation.get
import androidx.navigation.serialization.decodeArguments
import kotlinx.serialization.KSerializer
import kotlin.reflect.KClass
import kotlin.reflect.KType

typealias EntryTransitionScopeBuilder<T> = AnimatedContentTransitionScope<NavBackStackEntry>.() -> T

data class DestinationComposable<E, D: NavigationDestination<E>, GraphEvent>(
    val destinationClass: KClass<D>,
    val serializer: KSerializer<D>,
    val typeMap: Map<KType, @JvmSuppressWildcards NavType<*>> = emptyMap(),
    val deepLinks: List<NavDeepLink> = emptyList(),
    val enterTransition: EntryTransitionScopeBuilder<EnterTransition>? = null,
    val exitTransition: EntryTransitionScopeBuilder<ExitTransition>? = null,
    val popEnterTransition: EntryTransitionScopeBuilder<EnterTransition>? = enterTransition,
    val popExitTransition: EntryTransitionScopeBuilder<ExitTransition>? = exitTransition,
    override val eventHandler: ComponentEventHandler<E, D, GraphEvent> = ComponentEventHandler { destination, event, _, _ ->
        error(
            buildString {
                append("Unhandled Destination: ")
                append(destination)
                appendLine()
                append("Event: ")
                append(event)
            }
        )
    }
): NavigationComposable<E, D, GraphEvent> {

    fun NavGraphBuilder.composable(
        onNavigationEvent: (event: NavigationEvent) -> Unit,
        onGraphEvent: (GraphEvent) -> Unit
    ){
        destination(
            ComposeNavigatorDestinationBuilder(
                navigator = provider[ComposeNavigator::class],
                route = destinationClass,
                typeMap = typeMap,
                content = { entry ->
                    val destination = entry.getDestination()
                    destination.run {
                        Content(
                            entry = entry,
                            onEvent = {
                                eventHandler(
                                    component = destination,
                                    event = it,
                                    onNavigationEvent = onNavigationEvent,
                                    onEvent = onGraphEvent
                                )
                            }
                        )
                    }
                }
            ).apply {
                deepLinks.forEach { deepLink -> deepLink(deepLink) }
                enterTransition = this@DestinationComposable.enterTransition
                exitTransition = this@DestinationComposable.exitTransition
                popEnterTransition = this@DestinationComposable.popEnterTransition
                popExitTransition = this@DestinationComposable.popExitTransition
            }
        )
    }

    @SuppressLint("RestrictedApi")
    private fun NavBackStackEntry.getDestination(): D {
        val bundle = arguments ?: Bundle()
        val typeMap = destination.arguments.mapValues { it.value.type }
        return serializer.decodeArguments(bundle, typeMap)
    }
}

