package com.example.windowinsetstest.ui.navigation

import android.annotation.SuppressLint
import android.os.Bundle
import androidx.compose.animation.EnterTransition
import androidx.compose.animation.ExitTransition
import androidx.navigation.NavBackStackEntry
import androidx.navigation.NavDeepLink
import androidx.navigation.NavGraphBuilder
import androidx.navigation.compose.ComposeNavigator
import androidx.navigation.compose.ComposeNavigatorDestinationBuilder
import androidx.navigation.get
import androidx.navigation.serialization.decodeArguments
import kotlinx.serialization.KSerializer

data class DestinationObjectComposable<E, D: NavigationDestination<E>, GraphEvent>(
    override val component: D,
    val serializer: KSerializer<D>,
    val deepLinks: List<NavDeepLink> = emptyList(),
    val enterTransition: EntryTransitionScopeBuilder<EnterTransition>? = null,
    val exitTransition: EntryTransitionScopeBuilder<ExitTransition>? = null,
    val popEnterTransition: EntryTransitionScopeBuilder<EnterTransition>? = enterTransition,
    val popExitTransition: EntryTransitionScopeBuilder<ExitTransition>? = exitTransition,
    override val eventHandler: ComponentEventHandler<E, D, GraphEvent> = ComponentEventHandler { destination, event, _, _ ->
        error(
            buildString {
                append("Unhandled Destination: ")
                append(destination)
                appendLine()
                append("Event: ")
                append(event)
            }
        )
    }
): NavigationComposable.Object<E, D, GraphEvent> {

    fun NavGraphBuilder.composable(
        onNavigationEvent: (event: NavigationEvent) -> Unit,
        onGraphEvent: (GraphEvent) -> Unit
    ){
        destination(
            ComposeNavigatorDestinationBuilder(
                navigator = provider[ComposeNavigator::class],
                route = component::class,
                typeMap = emptyMap(),
                content = { entry ->
                    val destination = entry.getDestination()
                    destination.run {
                        Content(
                            entry = entry,
                            onEvent = {
                                eventHandler(
                                    component = destination,
                                    event = it,
                                    onNavigationEvent = onNavigationEvent,
                                    onEvent = onGraphEvent
                                )
                            }
                        )
                    }
                }
            ).apply {
                deepLinks.forEach { deepLink -> deepLink(deepLink) }
                enterTransition = this@DestinationObjectComposable.enterTransition
                exitTransition = this@DestinationObjectComposable.exitTransition
                popEnterTransition = this@DestinationObjectComposable.popEnterTransition
                popExitTransition = this@DestinationObjectComposable.popExitTransition
            }
        )
    }

    @SuppressLint("RestrictedApi")
    private fun NavBackStackEntry.getDestination(): D {
        val bundle = arguments ?: Bundle()
        val typeMap = destination.arguments.mapValues { it.value.type }
        return serializer.decodeArguments(bundle, typeMap)
    }
}