package com.example.windowinsetstest.ui.navigation

import android.util.Log
import androidx.navigation.NavController
import androidx.navigation.NavOptionsBuilder
import kotlin.reflect.KClass

sealed interface NavigationEvent {
    data class Navigate<T: NavigationDestination<*>>(
        val destination: T,
        val builder: NavOptionsBuilder.() -> Unit = {}
    ): NavigationEvent

    data object NavigateUp: NavigationEvent

    sealed interface PopBackStack: NavigationEvent {
        data class DestinationClass<T: Any>(
            val destinationClass: KClass<T>,
            val inclusive: Boolean,
            val saveState: Boolean = false
        ) : PopBackStack

        data class Destination<T : Any>(
            val destination: T,
            val inclusive: Boolean,
            val saveState: Boolean = false
        ) : PopBackStack
    }

    /**
     * [NavigationEvent] used for propagating Nested Events sequentially passed down through
     * [NavigationDestination] with [NavHost]
     *
     * [event] should be consumed by Primary [NavController] and
     * [nestedEvent] should be consumed by Nested [NavController] inside the Primary NavControllers NavHost
     */
    data class NestedNavigationEvent(
        val event: NavigationEvent,
        val nestedEvent: NavigationEvent
    ): NavigationEvent {
        init {
            require(event !is NestedNavigationEvent){
                "$event should not be a NestedNavigationEvent to avoid recursive calls"
            }
        }
    }

    companion object{
        fun NavController.onEvent(event: NavigationEvent){
            Log.i("Navigation", "Navigation Event: $event")
            when(event){
                is Navigate<*> -> navigate(event.destination, event.builder)
                NavigateUp -> navigateUp()
                is PopBackStack.Destination<*> -> popBackStack(
                    route = event.destination,
                    inclusive = event.inclusive,
                    saveState = event.saveState
                )
                is PopBackStack.DestinationClass<*> -> TODO()
                is NestedNavigationEvent -> onEvent(event.event)
            }
        }
    }
}