/***
 Copyright (c) 2017 CommonsWare, LLC
 Licensed under the Apache License, Version 2.0 (the "License"); you may not
 use this file except in compliance with the License. You may obtain	a copy
 of the License at http://www.apache.org/licenses/LICENSE-2.0. Unless required
 by applicable law or agreed to in writing, software distributed under the
 License is distributed on an "AS IS" BASIS,	WITHOUT	WARRANTIES OR CONDITIONS
 OF ANY KIND, either express or implied. See the License for the specific
 language governing permissions and limitations under the License.

 Covered in detail in the book _The Busy Coder's Guide to Android Development_
 https://commonsware.com/Android
 */

package com.commonsware.cwac.saferoom;

import android.arch.persistence.db.SimpleSQLiteQuery;
import android.arch.persistence.db.SupportSQLiteDatabase;
import android.arch.persistence.db.SupportSQLiteQuery;
import android.arch.persistence.db.SupportSQLiteStatement;
import android.arch.persistence.room.RoomSQLiteQuery;
import android.content.ContentValues;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteTransactionListener;
import android.os.CancellationSignal;
import android.util.Pair;
import net.sqlcipher.database.SQLiteCursor;
import net.sqlcipher.database.SQLiteCursorDriver;
import net.sqlcipher.database.SQLiteQuery;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Locale;

public class Database implements SupportSQLiteDatabase {
  private final net.sqlcipher.database.SQLiteDatabase safeDb;

  public Database(net.sqlcipher.database.SQLiteDatabase safeDb) {
    this.safeDb=safeDb;
  }

  @Override
  public SupportSQLiteStatement compileStatement(String sql) {
    return(new Statement(safeDb.compileStatement(sql)));
  }

  @Override
  public void beginTransaction() {
    safeDb.beginTransaction();
  }

  @Override
  public void beginTransactionNonExclusive() {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
  }

  @Override
  public void beginTransactionWithListener(
    SQLiteTransactionListener transactionListener) {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
  }

  @Override
  public void beginTransactionWithListenerNonExclusive(
    SQLiteTransactionListener transactionListener) {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
  }

  @Override
  public void endTransaction() {
    safeDb.endTransaction();
  }

  @Override
  public void setTransactionSuccessful() {
    safeDb.setTransactionSuccessful();
  }

  @Override
  public boolean inTransaction() {
    return(safeDb.inTransaction());
  }

  @Override
  public boolean isDbLockedByCurrentThread() {
    return(safeDb.isDbLockedByCurrentThread());
  }

  @Override
  public boolean yieldIfContendedSafely() {
    return(safeDb.yieldIfContendedSafely());
  }

  @Override
  public boolean yieldIfContendedSafely(long sleepAfterYieldDelay) {
    return(safeDb.yieldIfContendedSafely(sleepAfterYieldDelay));
  }

  @Override
  public int getVersion() {
    return(safeDb.getVersion());
  }

  @Override
  public void setVersion(int version) {
    safeDb.setVersion(version);
  }

  @Override
  public long getMaximumSize() {
    return(safeDb.getMaximumSize());
  }

  @Override
  public long setMaximumSize(long numBytes) {
    return(safeDb.setMaximumSize(numBytes));
  }

  @Override
  public long getPageSize() {
    return(safeDb.getPageSize());
  }

  @Override
  public void setPageSize(long numBytes) {
    safeDb.setPageSize(numBytes);
  }

  @Override
  public Cursor query(String sql) {
    return(query(new SimpleSQLiteQuery(sql)));
  }

  @Override
  public Cursor query(String sql, Object[] selectionArgs) {
    return(query(new SimpleSQLiteQuery(sql, selectionArgs)));
  }

  @Override
  public Cursor query(final SupportSQLiteQuery supportQuery) {
    return(query(supportQuery, null));
  }

  @Override
  public Cursor query(final SupportSQLiteQuery supportQuery,
                      CancellationSignal signal) {
    int count=0;

    try {
      if (supportQuery instanceof RoomSQLiteQuery) {
        Field argCount = RoomSQLiteQuery.class.getDeclaredField("mArgCount");
        argCount.setAccessible(true);
        count = argCount.getInt(supportQuery);
      }
      else if (supportQuery instanceof SimpleSQLiteQuery) {
        Field bindArgs = SimpleSQLiteQuery.class.getDeclaredField("mBindArgs");
        bindArgs.setAccessible(true);
        Object[] bindArgsValue = (Object[]) bindArgs.get(supportQuery);
        count = bindArgsValue!=null?bindArgsValue.length:0;
      }
    }
    catch (Exception e) {
      throw new IllegalStateException("Um, ick", e);
    }

    String[] fakeArgs=new String[count];

    for (int i=0;i<count;i++) {
      fakeArgs[i]="";
    }

    return(safeDb.rawQueryWithFactory(
      new net.sqlcipher.database.SQLiteDatabase.CursorFactory() {
        @Override
        public net.sqlcipher.Cursor newCursor(
          net.sqlcipher.database.SQLiteDatabase db,
          SQLiteCursorDriver masterQuery, String editTable,
          SQLiteQuery query) {
          supportQuery.bindTo(new Program(query));
          return new SQLiteCursor(db, masterQuery, editTable, query);
        }
      }, supportQuery.getSql(), fakeArgs, null));
  }

  @Override
  public long insert(String table, int conflictAlgorithm,
                     ContentValues values) {
    return(safeDb.insertWithOnConflict(table, null, values, conflictAlgorithm));
  }

  @Override
  public int delete(String table, String whereClause, Object[] whereArgs) {
    return(safeDb.delete(table, whereClause, stringify(whereArgs)));
  }

  @Override
  public int update(String table, int conflictAlgorithm, ContentValues values,
                    String whereClause, Object[] whereArgs) {
    return(safeDb.updateWithOnConflict(table, values, whereClause,
      stringify(whereArgs), conflictAlgorithm));
  }

  @Override
  public void execSQL(String sql) throws SQLException {
    safeDb.execSQL(sql);
  }

  @Override
  public void execSQL(String sql, Object[] bindArgs) throws SQLException {
    safeDb.execSQL(sql, bindArgs);
  }

  @Override
  public boolean isReadOnly() {
    return(safeDb.isReadOnly());
  }

  @Override
  public boolean isOpen() {
    return(safeDb.isOpen());
  }

  @Override
  public boolean needUpgrade(int newVersion) {
    return(safeDb.needUpgrade(newVersion));
  }

  @Override
  public String getPath() {
    return(safeDb.getPath());
  }

  @Override
  public void setLocale(Locale locale) {
    safeDb.setLocale(locale);
  }

  @Override
  public void setMaxSqlCacheSize(int cacheSize) {
    safeDb.setMaxSqlCacheSize(cacheSize);
  }

  @Override
  public void setForeignKeyConstraintsEnabled(boolean enable) {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
//    safeDb.setForeignKeyConstraintsEnabled(enable);
  }

  @Override
  public boolean enableWriteAheadLogging() {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
//    return(safeDb.enableWriteAheadLogging());
  }

  @Override
  public void disableWriteAheadLogging() {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
//    safeDb.disableWriteAheadLogging();
  }

  @Override
  public boolean isWriteAheadLoggingEnabled() {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
//    return(safeDb.isWriteAheadLoggingEnabled());
  }

  @Override
  public List<Pair<String, String>> getAttachedDbs() {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
//    return(safeDb.getAttachedDbs());
  }

  @Override
  public boolean isDatabaseIntegrityOk() {
    // TODO not supported in SQLCipher for Android
    throw new UnsupportedOperationException("I kinna do it, cap'n!");
//    return(safeDb.isDatabaseIntegrityOk());
  }

  @Override
  public void close() {
    safeDb.close();
  }

  private String[] stringify(Object[] params) {
    String[] result=new String[params.length];

    for (int i=0;i<params.length;i++) {
      result[i]=params[i].toString();
    }

    return(result);
  }
}
