package com.genimee.android.utils.threads

import android.os.Process
import java.util.concurrent.BlockingQueue
import java.util.concurrent.ThreadFactory
import java.util.concurrent.ThreadPoolExecutor
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicInteger

class ScalingThreadPoolExecutor : ThreadPoolExecutor {
    /**
     * Number of threads that are actively executing tasks.
     */
    private val activeCount = AtomicInteger()
    private var defaultThreadPriority = Process.THREAD_PRIORITY_BACKGROUND

    private constructor(corePoolSize: Int, maximumPoolSize: Int, keepAliveTime: Long, unit: TimeUnit, workQueue: BlockingQueue<Runnable>)
            : super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue)

    private constructor(corePoolSize: Int, maximumPoolSize: Int, keepAliveTime: Long, unit: TimeUnit, workQueue: BlockingQueue<Runnable>, threadFactory: ThreadFactory)
            : super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory)

    override fun getActiveCount(): Int = activeCount.get()

    override fun beforeExecute(thread: Thread, runnable: Runnable?) {
        try {
            thread.priority = defaultThreadPriority
        } catch (ignore: Exception) {
        }
        activeCount.incrementAndGet()
    }

    override fun afterExecute(runnable: Runnable?, throwable: Throwable?) {
        activeCount.decrementAndGet()
    }

    companion object {

        @JvmStatic
        @JvmOverloads
        fun newScalingThreadPool(min: Int, max: Int, keepAliveTime: Long, groupName: String?, defaultPriority: Int = Process.THREAD_PRIORITY_BACKGROUND): ThreadPoolExecutor {
            val queue = ScalingQueue<Runnable>()
            return if (groupName.isNullOrEmpty()) {
                ScalingThreadPoolExecutor(min, max, keepAliveTime, TimeUnit.SECONDS, queue)
            } else {
                ScalingThreadPoolExecutor(min, max, keepAliveTime, TimeUnit.SECONDS, queue, GroupNameThreadFactory(groupName ?: "default"))
            }.apply {
                rejectedExecutionHandler = ForceQueuePolicy()
                defaultThreadPriority = defaultPriority
                queue.setThreadPoolExecutor(this)
            }
        }
    }
}
